/*
 * Copyright 2007 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*-
 * Copyright (c) 2003
 *	Bill Paul <wpaul@windriver.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/dditypes.h>
#include <sys/stream.h>
#include <sys/termio.h>
#include <sys/disp.h>
#include <sys/thread.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/cmn_err.h>
#include <sys/stropts.h>
#include <sys/strtty.h>
#include <sys/kbio.h>
#include <sys/cred.h>
#include <sys/stat.h>
#include <sys/consdev.h>
#include <sys/kmem.h>
#include <sys/modctl.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>
#include <sys/pci.h>
#include <sys/errno.h>
#include <sys/gld.h>
#include <sys/dlpi.h>
#include <sys/ethernet.h>
#include <inet/common.h>
#include <inet/nd.h>
#include <inet/mi.h>
#include <sys/strsun.h>
#include <sys/spl.h>

#include <queue.h>
#include <if_ieee80211.h>

#include <pe_var.h>
#include <resource_var.h>
#include <ntoskrnl_var.h>
#include <ndis_var.h>
#include <hal_var.h>
#include <cfg_var.h>
#include <if_ndisvar.h>

uint32_t ndis_debug;
#define NDIS_DUMMY_PATH "\\\\some\\bogus\\path"

static unsigned char dbbuffer[50];

__stdcall static void ndis_status_func(ndis_handle, ndis_status,
	void *, uint32_t);
__stdcall static void ndis_statusdone_func(ndis_handle);
__stdcall static void ndis_setdone_func(ndis_handle, ndis_status);
__stdcall static void ndis_getdone_func(ndis_handle, ndis_status);
__stdcall static void ndis_resetdone_func(ndis_handle, ndis_status, uint8_t);
__stdcall static void ndis_sendrsrcavail_func(ndis_handle);
__stdcall static void fake_intrhand(kdpc *dpc, device_object *dobj, irp *ip,  struct ndis_softc *sc);
void ndis_intrhandler(void *arg);

static image_patch_table kernndis_functbl[] = {
	IMPORT_FUNC(ndis_status_func),
	IMPORT_FUNC(ndis_statusdone_func),
	IMPORT_FUNC(ndis_setdone_func),
	IMPORT_FUNC(ndis_getdone_func),
	IMPORT_FUNC(ndis_resetdone_func),
	IMPORT_FUNC(ndis_sendrsrcavail_func),
	{ NULL, NULL, NULL }
};

struct nd_head ndis_devhead;

struct ndis_req {
	void			(*nr_func)(void *);
	void			*nr_arg;
	int			nr_exit;
	STAILQ_ENTRY(ndis_req)	link;
};

struct ndisproc {
	struct ndisqhead	*np_q;
	kcondvar_t		*cv_p;
	kmutex_t		*mtx_p;
	int			np_state;
	int			cnt;
	struct ndis_req 	*np_curjob;
};

static void ndis_return(void *);
static int ndis_create_kthreads(void);
static void ndis_destroy_kthreads(void);
static void ndis_stop_thread(int);
static int ndis_enlarge_thrqueue(int);
static int ndis_shrink_thrqueue(int);
static void ndis_runq(void *);

kmutex_t   ndis_req_mtx;
kmutex_t   ndis_thr_mtx1;
kmutex_t   ndis_thr_mtx2;
kmutex_t   task_queue_mtx;
kmutex_t   set_mtx,get_mtx,reset_mtx;
kcondvar_t ndis_cv1,ndis_cv2;
kcondvar_t set_get_cvp,reset_cvp;

static STAILQ_HEAD(ndisqhead, ndis_req) ndis_ttodo;
static struct ndisqhead ndis_itodo;
static struct ndisqhead ndis_free;
static int ndis_jobs = 64;

static struct ndisproc ndis_tproc;
static struct ndisproc ndis_iproc;
int *ndis_i_pcnt = &(ndis_iproc.cnt);

/*
 * We create two kthreads for the NDIS subsystem. One of them is a task
 * queue for performing various odd jobs. The other is an swi thread
 * reserved exclusively for running interrupt handlers. The reason we
 * have our own task queue is that there are some cases where we may
 * need to sleep for a significant amount of time, and if we were to
 * use one of the taskqueue threads, we might delay the processing
 * of other pending tasks which might need to run right away. We have
 * a separate swi thread because we don't want our interrupt handling
 * to be delayed either.
 *
 * By default there are 32 jobs available to start, and another 8
 * are added to the free list each time a new device is created.
 */

static void
ndis_runq(arg)
	void			*arg;
{
	struct ndis_req		*r = NULL, *die = NULL;
	struct ndisproc		*p;
	int			i;
	int			s;

	p = arg;

	while (1) {

		/* Sleep, but preserve our original priority. */
		ndis_thsuspend(p->cv_p, p->mtx_p, 0);
		/* Look for any jobs on the work queue. */
		mutex_enter(&task_queue_mtx);
		p->np_state = NDIS_PSTATE_RUNNING;
		while(STAILQ_FIRST(p->np_q) != NULL) {
			r = STAILQ_FIRST(p->np_q);
			p->np_curjob = r;
			STAILQ_REMOVE_HEAD(p->np_q, link);
			p->cnt -= 0x1; 
			mutex_exit(&task_queue_mtx);
			/* Do the work. */
			if (r->nr_func != NULL)
				(*r->nr_func)(r->nr_arg);
			mutex_enter(&task_queue_mtx);
			STAILQ_INSERT_HEAD(&ndis_free, r, link);

			/* Check for a shutdown request */

			if (r->nr_exit == TRUE)
				die = r;
		}
		p->np_curjob = NULL;
		p->np_state = NDIS_PSTATE_SLEEPING;
		mutex_exit(&task_queue_mtx);

		/* Bail if we were told to shut down. */

		if (die != NULL)
			break;
	}
	thread_exit();
	return; /* notreached */
}

static int
ndis_create_kthreads()
{
	struct ndis_req		*r;
	int			i, error = 0;
	kthread_t	*pt1,*pt2;
	mutex_init(&ndis_thr_mtx1, "NDIS thread lock 1",
	   MUTEX_DEFAULT, NULL);
	mutex_init(&ndis_thr_mtx2, "NDIS thread lock 2",
	   MUTEX_DEFAULT, NULL);
	mutex_init(&task_queue_mtx, "NDIS queue lock", 
	   MUTEX_DEFAULT, NULL); 
	mutex_init(&set_mtx, "dpc lock mtx",
	   MUTEX_DEFAULT, NULL);
	mutex_init(&get_mtx, "dpc lock mtx",
	   MUTEX_DEFAULT, NULL);
	mutex_init(&reset_mtx, "reset lock mtx",
	   MUTEX_DEFAULT, NULL);
	mutex_init(&ndis_req_mtx, "req lock mtx",
	   MUTEX_DEFAULT, NULL);
	cv_init(&ndis_cv1, "NDIS task queue lock", CV_DEFAULT, NULL);
	cv_init(&ndis_cv2, "NDIS SWI queue lock", CV_DEFAULT, NULL);
	cv_init(&set_get_cvp, "dpc lock mtx", CV_DEFAULT, NULL);
	cv_init(&reset_cvp, "reset lock mtx", CV_DEFAULT, NULL);
	
	STAILQ_INIT(&ndis_ttodo);
	STAILQ_INIT(&ndis_itodo);
	STAILQ_INIT(&ndis_free);
	for (i = 0; i < ndis_jobs; i++) {
		r = kmem_zalloc(sizeof (struct ndis_req), KM_NOSLEEP);
		if (r == NULL) {
			error = ENOMEM;
			break;
		}
		STAILQ_INSERT_HEAD(&ndis_free, r, link);
	}

	if (error == 0) {
		ndis_tproc.np_q = &ndis_ttodo;
		ndis_tproc.np_state = NDIS_PSTATE_SLEEPING;
		pt1 = thread_create(NULL, NDIS_KSTACK_PAGES, ndis_runq, &ndis_tproc, 0, &p0, TS_RUN, 99);
		ndis_tproc.mtx_p = &ndis_thr_mtx1;
		ndis_tproc.cv_p = &ndis_cv1;
	}
	if (error == 0) {
		ndis_iproc.np_q = &ndis_itodo;
		ndis_iproc.np_state = NDIS_PSTATE_SLEEPING;
		pt2 = thread_create(NULL, NDIS_KSTACK_PAGES, ndis_runq, &ndis_iproc, 0, &p0, TS_RUN, 99);
		ndis_iproc.mtx_p = &ndis_thr_mtx2;
		ndis_iproc.cv_p = &ndis_cv2;
	}
	if (error) {
		while ((r = STAILQ_FIRST(&ndis_free)) != NULL) {
			STAILQ_REMOVE_HEAD(&ndis_free, link);
			kmem_free(r,sizeof (struct ndis_req));
	}
		return(error);
	}

	return(0);
}

static void
ndis_destroy_kthreads()
{
	struct ndis_req		*r;

	/* Stop the threads. */

	ndis_stop_thread(NDIS_TASKQUEUE);
	ndis_stop_thread(NDIS_SWI);

	/* Destroy request structures. */

	while ((r = STAILQ_FIRST(&ndis_free)) != NULL) {
		STAILQ_REMOVE_HEAD(&ndis_free, link);
		kmem_free(r,sizeof (struct ndis_req));
	}
	mutex_destroy(&ndis_thr_mtx1);
	mutex_destroy(&ndis_thr_mtx2);
	mutex_destroy(&task_queue_mtx);
	mutex_destroy(&set_mtx);
	mutex_destroy(&get_mtx);
	mutex_destroy(&ndis_req_mtx);
	cv_destroy(&ndis_cv1);
	cv_destroy(&ndis_cv2);
	cv_destroy(&set_get_cvp);
	cv_destroy(&reset_cvp);
	return;
}

static void
ndis_stop_thread(t)
	int			t;
{
	struct ndis_req		*r;
	struct ndisqhead	*q;
	kcondvar_t	*p;

	if (t == NDIS_TASKQUEUE) {
		q = &ndis_ttodo;
		p = ndis_tproc.cv_p;
	} else {
		q = &ndis_itodo;
		p = ndis_iproc.cv_p;
	}

	/* Create and post a special 'exit' job. */

	mutex_enter(&task_queue_mtx);
	r = STAILQ_FIRST(&ndis_free);
	STAILQ_REMOVE_HEAD(&ndis_free, link);
	r->nr_func = NULL;
	r->nr_arg = NULL;
	r->nr_exit = TRUE;
	STAILQ_INSERT_TAIL(q, r, link);
	mutex_exit(&task_queue_mtx);

	ndis_thresume(p);

	/* wait for thread exit 60HZ */

	delay(60);

	/* Now empty the job list. */

	mutex_enter(&task_queue_mtx);
	while ((r = STAILQ_FIRST(q)) != NULL) {
		STAILQ_REMOVE_HEAD(q, link);
		STAILQ_INSERT_HEAD(&ndis_free, r, link);
	}
	mutex_exit(&task_queue_mtx);

	return;
}

static int
ndis_enlarge_thrqueue(cnt)
	int			cnt;
{
	struct ndis_req		*r;
	int			i;

	for (i = 0; i < cnt; i++) {
		r = kmem_zalloc(sizeof (struct ndis_req), KM_NOSLEEP);
		if (r == NULL)
			return(ENOMEM);
		mutex_enter(&task_queue_mtx);
		STAILQ_INSERT_HEAD(&ndis_free, r, link);
		ndis_jobs++;
		mutex_exit(&task_queue_mtx);
	}

	return(0);
}

static int
ndis_shrink_thrqueue(cnt)
	int			cnt;
{
	struct ndis_req		*r;
	int			i;

	for (i = 0; i < cnt; i++) {
		r = kmem_zalloc(sizeof (struct ndis_req), KM_NOSLEEP);
		if (r == NULL)
			return(ENOMEM);
		mutex_enter(&task_queue_mtx);
		STAILQ_INSERT_HEAD(&ndis_free, r, link);
		ndis_jobs++;
		mutex_exit(&task_queue_mtx);
	}

	return(0);
}

int
ndis_unsched(func, arg, t)
	void			(*func)(void *);
	void			*arg;
	int			t;
{
	struct ndis_req		*r;
	struct ndisqhead	*q;

	if (t == NDIS_TASKQUEUE) {
		q = &ndis_ttodo;
	} else {
		q = &ndis_itodo;
	}

	mutex_enter(&task_queue_mtx);
	STAILQ_FOREACH(r, q, link) {
		if (r->nr_func == func && r->nr_arg == arg) {
			STAILQ_REMOVE(q, r, ndis_req, link);
			if(q == &ndis_itodo)
				ndis_iproc.cnt -= 0x1;
			else
				ndis_tproc.cnt -= 0x1;
			STAILQ_INSERT_HEAD(&ndis_free, r, link);
			mutex_exit(&task_queue_mtx);
			return(0);
		}
	}

	mutex_exit(&task_queue_mtx);

	return(ENOENT);
}

int
ndis_sched(func, arg, t)
	void			(*func)(void *);
	void			*arg;
	int			t;
{
	struct ndis_req		*r;
	struct ndisqhead	*q;
	kcondvar_t	*p;
	int			s;
	
	mutex_enter(&task_queue_mtx);

	if (t == NDIS_TASKQUEUE) {
		q = &ndis_ttodo;
		p = ndis_tproc.cv_p;
	} else {
		q = &ndis_itodo;
		p = ndis_iproc.cv_p;
	}
	/*
	 * Check to see if an instance of this job is already
	 * pending. If so, don't bother queuing it again.
	 */
	STAILQ_FOREACH(r, q, link) {
		if (r->nr_func == func && r->nr_arg == arg) {
			mutex_exit(&task_queue_mtx);
			return(0);
		}
	}

	r = STAILQ_FIRST(&ndis_free);
	if (r == NULL) {
		r = kmem_zalloc(sizeof (struct ndis_req), KM_NOSLEEP);
		if(!r)
			panic("So, no way!\n");
		cmn_err(CE_CONT,"ndis_free empty!!!\n");
		goto skip;
	}
			
	STAILQ_REMOVE_HEAD(&ndis_free, link);
skip:
	r->nr_func = func;
	r->nr_arg = arg;
	r->nr_exit = FALSE;
	STAILQ_INSERT_TAIL(q, r, link);
	if (t == NDIS_TASKQUEUE){
		s = ndis_tproc.np_state;
		ndis_tproc.cnt += 0x1;
	}else{
		s = ndis_iproc.np_state;
		ndis_iproc.cnt += 0x1;
	}

	/*
	 * Post the job, but only if the thread is actually blocked
	 * on its own suspend call. If a driver queues up a job with
	 * NdisScheduleWorkItem() which happens to do a KeWaitForObject(),
	 * it may suspend there, and in that case we don't want to wake
	 * it up until KeWaitForObject() gets woken up on its own.
	 */
	if (s == NDIS_PSTATE_SLEEPING)
		ndis_thresume(p);

	mutex_exit(&task_queue_mtx);
	return(0);
}

int
ndis_thsuspend(cv_p, mtx_p, timo)
	kcondvar_t		*cv_p;
	kmutex_t		*mtx_p;
	int			timo;
{
	clock_t err=0;
	
	mutex_enter(mtx_p);
	if(timo)
		err = cv_timedwait(cv_p, mtx_p, timo + ddi_get_lbolt());
	else{
		cv_wait(cv_p,mtx_p);
		err = 0;	
	}
	mutex_exit(mtx_p);
	return(err);
}

void
ndis_thresume(cv_p)
	kcondvar_t	*cv_p;
{
	cv_signal(cv_p);
	return;
}

__stdcall static void
ndis_sendrsrcavail_func(adapter)
	ndis_handle		adapter;
{
	return;
}

__stdcall static void
ndis_status_func(adapter, status, sbuf, slen)
	ndis_handle		adapter;
	ndis_status		status;
	void			*sbuf;
	uint32_t		slen;
{
//	cmn_err(CE_CONT, "status: %x\n", status);
	return;
}

__stdcall static void
ndis_statusdone_func(adapter)
	ndis_handle		adapter;
{
//	cmn_err(CE_CONT, "status done, Okay\n");
	return;
}

__stdcall static void
ndis_setdone_func(adapter, status)
	ndis_handle		adapter;
	ndis_status		status;
{
	ndis_miniport_block	*block;
	block = adapter;

	block->nmb_setstat = status;
	cmn_err(CE_CONT,"setdone");
	cv_signal(&set_get_cvp);  /*fixme*/
	return;
}

__stdcall static void
ndis_getdone_func(adapter, status)
	ndis_handle		adapter;
	ndis_status		status;
{
	ndis_miniport_block	*block;
	block = adapter;

	block->nmb_getstat = status;
	cmn_err(CE_CONT,"getdone");
	cv_signal(&set_get_cvp);
	return;
}

__stdcall static void
ndis_resetdone_func(adapter, status, addressingreset)
	ndis_handle		adapter;
	ndis_status		status;
	uint8_t			addressingreset;
{
	cmn_err(CE_CONT, "reset done...\n");
	cv_signal(&reset_cvp); 
	return;
}

int
ndis_create_sysctls(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_cfg		*vals;
	struct ndis_cfglist		*each;
	char			buf[256];
	uint32_t error = 0x1;

	if (arg == NULL)
		return(EINVAL);

	sc = (struct ndis_softc *)arg;

	TAILQ_INIT(&sc->ndis_cfglist_head);
	/* Add the driver-specific registry keys. */

	vals =(ndis_cfg *) sc->ndis_regvals;
	while(1) {
		if (vals->nc_cfgkey == NULL)
			break;
		if (vals->nc_idx != sc->ndis_devidx) {
			vals++;
			continue;
		}

		/* See if we already have a sysctl with this name */
		TAILQ_FOREACH(each, &sc->ndis_cfglist_head, link) {
			if ((error = ndis_strcasecmp(each->ndis_cfg.nc_cfgkey, vals->nc_cfgkey)) == 0)
				break;
		}

		if(error){
			ndis_add_sysctl(arg, vals->nc_cfgkey, vals->nc_cfgdesc, vals->nc_val);
			
		}

		vals++;
	}

	/*
	 * Environment can be either Windows (0) or WindowsNT (1).
	 * We qualify as the latter.
	 */
	ndis_add_sysctl(sc, "Environment",
	    "Windows environment", "1");

	/* NDIS version should be 5.1. */
	ndis_add_sysctl(sc, "NdisVersion",
	    "NDIS API Version", "0x00050001");

	/* Bus type (PCI, PCMCIA, etc...) */
	sprintf(buf, "%d", (int)sc->ndis_iftype);
	ndis_add_sysctl(sc, "BusType", "Bus Type", buf);

	/* Base I/O address */	
	sprintf(buf, "0xfafec000");
	ndis_add_sysctl(sc, "IOBaseAddress",
	    "Base I/O Address", buf);
	
	return(0);
}

char *
strdup(const char *s)
{
	size_t len;
	char *new;

	len = strlen(s);
	new = kmem_zalloc(len + 1, KM_NOSLEEP);
	bcopy(s, new, len);
	new[len] = '\0';
	return (new);
}

int
ndis_add_sysctl(arg, key, desc, val)
	void			*arg;
	char			*key;
	char			*desc;
	char			*val;
{
	struct ndis_softc	*sc;
	struct ndis_cfglist	*cfg;
	char			descstr[256];

	sc = arg;

	cfg = kmem_zalloc(sizeof(struct ndis_cfglist), KM_NOSLEEP);

	if (cfg == NULL)
		return(ENOMEM);

	cfg->ndis_cfg.nc_cfgkey = strdup(key);
	if (desc == NULL) {
		snprintf(descstr, sizeof(descstr), "%s (dynamic)", key);
		cfg->ndis_cfg.nc_cfgdesc = strdup(descstr);
	} else
		cfg->ndis_cfg.nc_cfgdesc = strdup(desc);
	strcpy(cfg->ndis_cfg.nc_val, val);
	cfg->ndis_cfg.nc_idx = sc->ndis_devidx;
	
	TAILQ_INSERT_TAIL(&sc->ndis_cfglist_head, cfg, link);
	return(0);
}

#define CFGVAR each->ndis_cfg

int
ndis_dump_sysctls(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	struct ndis_cfglist	*cfg,*each;
	int			i = 0;
	sc = arg;

	if (sc == NULL)
		return (EINVAL);
	TAILQ_FOREACH(each, &sc->ndis_cfglist_head, link) {
		CFGVAR.nc_val[255] = '\0';
		cmn_err(CE_CONT,"%d : nc_cfgkey :%s\n",i,each->ndis_cfg.nc_cfgkey);
		cmn_err(CE_CONT,"%d : nc_cfgdesc:%s\n",i,each->ndis_cfg.nc_cfgdesc);
		cmn_err(CE_CONT,"%d : nc_val    :%s\n",i,CFGVAR.nc_val);
		cmn_err(CE_CONT,"%d : nc_idx    :%d\n",i,each->ndis_cfg.nc_idx);
		i++;
	}

	return(0);
}

int
ndis_flush_sysctls(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	struct ndis_cfglist	*cfg;

	sc = arg;

	while (!TAILQ_EMPTY(&sc->ndis_cfglist_head)) {
		cfg = TAILQ_FIRST(&sc->ndis_cfglist_head);
		TAILQ_REMOVE(&sc->ndis_cfglist_head, cfg, link);
		if(cfg->ndis_cfg.nc_cfgkey && strlen(cfg->ndis_cfg.nc_cfgkey))
			kmem_free(cfg->ndis_cfg.nc_cfgkey, 
				strlen(cfg->ndis_cfg.nc_cfgkey)+1);
		if(cfg->ndis_cfg.nc_cfgdesc &&  strlen(cfg->ndis_cfg.nc_cfgdesc))
			kmem_free(cfg->ndis_cfg.nc_cfgdesc, 
				strlen(cfg->ndis_cfg.nc_cfgdesc)+1);
		if(cfg)
			kmem_free(cfg, sizeof(struct ndis_cfglist));
	}

	return(0);
}

static void
ndis_return(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	__stdcall ndis_return_handler	returnfunc;
	ndis_handle		adapter;
	ndis_packet		*p;
	uint8_t			irql;

	p = arg;
	sc = p->np_softc;
	adapter = sc->ndis_block->nmb_miniportadapterctx;

	if (adapter == NULL)
		return;

	returnfunc = sc->ndis_chars->nmc_return_packet_func;

//	KeAcquireSpinLock(&sc->ndis_block->nmb_lock, &irql);
	MSCALL2(returnfunc, adapter, p);
//	KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);

	return;
}

void
ndis_return_packet(buf, arg)
	void			*buf;	/* not used */
	void			*arg;
{
	ndis_packet		*p;

	if (arg == NULL)
		return;

	p = arg;

	/* Decrement refcount. */
	p->np_refcnt--;

	/* Release packet when refcount hits zero, otherwise return. */
	if (p->np_refcnt){
		cmn_err(CE_CONT,"return_packet failed\n");
		return;
	}
	ndis_return(p);
//	ndis_sched(ndis_return, p, NDIS_TASKQUEUE);
	return;
}

void
ndis_free_bufs(b0)
	ndis_buffer		*b0;
{
	ndis_buffer		*next;

	if (b0 == NULL)
		return;

	while(b0 != NULL) {
		next = b0->mdl_next;
		IoFreeMdl(b0);
		b0 = next;
	}

	return;
}

void
ndis_free_packet(p)
	ndis_packet		*p;
{
	if (p == NULL)
		return;
	ndis_free_bufs(p->np_private.npp_head);
	NdisFreePacket(p);
	return;
}

uint32_t phyaddsize=0;
ndis_physaddr ndis_iostart[6];
uint32_t ndis_iolength[6];

int
ndis_convert_res(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_resource_list	*rl = NULL;
	cm_partial_resource_desc	*prd = NULL;
	ndis_miniport_block	*block;
	int  error = 0;
	uint32_t nb;
	pci_regspec_t *pci_regp;
	int i=0;
	
	sc = arg;
	block = sc->ndis_block;

	if (ddi_prop_lookup_int_array(DDI_DEV_T_ANY, sc->devinfo_this, DDI_PROP_DONTPASS,
		"assigned-addresses", (int **)&pci_regp, &nb) != DDI_SUCCESS) {
		return(DDI_FAILURE);
	}
	sc->ndis_rescnt = nb/(sizeof(pci_regspec_t)/sizeof(uint32_t)) + 1;
	sc->ndis_block->nmb_rlist_size = sizeof(ndis_resource_list) +
	    (sizeof(cm_partial_resource_desc) * (sc->ndis_rescnt - 1));
	rl = kmem_zalloc(sc->ndis_block->nmb_rlist_size, KM_NOSLEEP);

	if (rl == NULL) {
		ddi_prop_free(pci_regp);
		return(ENOMEM);
	}

	rl->cprl_version = 5;
	rl->cprl_version = 1;
	rl->cprl_count = sc->ndis_rescnt;
	prd = rl->cprl_partial_descs;
	for(i = 0; i < (nb / (sizeof(pci_regspec_t)/sizeof(uint32_t))); i++){
		switch (pci_regp[i].pci_phys_hi & PCI_REG_ADDR_M) {
			case PCI_ADDR_IO:
				prd->cprd_type = CmResourceTypePort;
				prd->cprd_flags = CM_RESOURCE_PORT_IO;
				prd->cprd_sharedisp = CmResourceShareDeviceExclusive;
				prd->ut.cprd_port.cprd_start.np_quad = 
						pci_regp[i].pci_phys_low;
				prd->ut.cprd_port.cprd_len = 
						pci_regp[i].pci_size_low;
				cmn_err(CE_CONT,
					"I/O PhysAddr:%x, Size:%d",
					pci_regp[i].pci_phys_low,
					pci_regp[i].pci_size_low);
				phyaddsize=pci_regp->pci_size_low;
				prd++;
				break;
			case PCI_ADDR_MEM32:
			case PCI_ADDR_MEM64:
				prd->cprd_type = CmResourceTypeMemory;
				prd->cprd_flags = CM_RESOURCE_MEMORY_READ_WRITE;
				prd->cprd_sharedisp = CmResourceShareDeviceExclusive;
				prd->ut.cprd_port.cprd_start.np_quad = 
						pci_regp[i].pci_phys_low;
				prd->ut.cprd_port.cprd_len = 
						pci_regp[i].pci_size_low;
				cmn_err(CE_CONT,
					"Mem PhysAddr:0x%x, Size:0x%x\n",
					pci_regp[i].pci_phys_low,
					pci_regp[i].pci_size_low);
				ndis_iostart[i].np_quad = prd->ut.cprd_port.cprd_start.np_quad;
				ndis_iolength[i] = pci_regp[i].pci_size_low;
				prd++;
				break;
			default:
				panic("So bad, unknown register type\n");
				break;
		}
	}
	prd->cprd_type = CmResourceTypeInterrupt;
	prd->cprd_flags = 0;
	prd->cprd_sharedisp = CmResourceShareShared;
	prd->ut.cprd_intr.cprd_level = 6;
	prd->ut.cprd_intr.cprd_vector = 5;
	prd->ut.cprd_intr.cprd_affinity = 0;

	block->nmb_rlist = rl;
	ddi_prop_free(pci_regp);
	return(error);
}


/*
 * Map an NDIS packet to an mbuf list. When an NDIS driver receives a
 * packet, it will hand it to us in the form of an ndis_packet,
 * which we need to convert to an mbuf that is then handed off
 * to the stack. Note: we configure the mbuf list so that it uses
 * the memory regions specified by the ndis_buffer structures in
 * the ndis_packet as external storage. In most cases, this will
 * point to a memory region allocated by the driver (either by
 * ndis_malloc_withtag() or ndis_alloc_sharedmem()). We expect
 * the driver to handle free()ing this region for is, so we set up
 * a dummy no-op free handler for it.
 */ 

int
ndis_ptom(m0, p)
	mblk_t			**m0;
	ndis_packet		*p;
{
	mblk_t			*m, *head, *prev = NULL;
	ndis_buffer		*buf;
	ndis_packet_private	*priv;
	uint32_t		totlen = 0;
	uint32_t		len = 0;
	uint32_t		i = 0;
	
	if (p == NULL || m0 == NULL)
		return(EINVAL);

	head = *m0;
	priv = &p->np_private;
	buf = priv->npp_head;
	p->np_refcnt = 0;
	
	for (buf = priv->npp_head; buf != NULL; buf = buf->mdl_next) {
		m = allocb(MmGetMdlByteCount(buf), BPRI_MED);
		if (m == NULL) {
			freeb(*m0);
			*m0 = NULL;
			return(ENOBUFS);
		}
		len = MmGetMdlByteCount(buf);
		bcopy(MmGetMdlVirtualAddress(buf), m->b_wptr, len);
		m->b_wptr = m->b_rptr + len;
		p->np_refcnt++;

		if (head == *m0)
			*m0 = m;
		else{
			prev->b_cont = m;
			cmn_err(CE_CONT,"%s, continue\n",__func__);
		}
		prev = m;
	}
	ndis_return_packet(buf,p);
	return(0);
}

/*
 * Create an mbuf chain from an NDIS packet chain.
 * This is used mainly when transmitting packets, where we need
 * to turn an mbuf off an interface's send queue and transform it
 * into an NDIS packet which will be fed into the NDIS driver's
 * send routine.
 *
 * NDIS packets consist of two parts: an ndis_packet structure,
 * which is vaguely analagous to the pkthdr portion of an mbuf,
 * and one or more ndis_buffer structures, which define the
 * actual memory segments in which the packet data resides.
 * We need to allocate one ndis_buffer for each mbuf in a chain,
 * plus one ndis_packet as the header.
 */

int
ndis_mtop(m0, p)
	mblk_t			*m0;
	ndis_packet		**p;
{
	mblk_t			*m, *t;
	ndis_buffer		*buf = NULL, *prev = NULL;
	ndis_packet_private	*priv;
	uint32_t		len;
	char			*real_buf;
	int			i;

	if (p == NULL || *p == NULL || m0 == NULL){
		cmn_err(CE_CONT,"%s:mistake\n",__func__);
		return(EINVAL);
	}
	priv = &(*p)->np_private;
	if(priv->npp_head){
		ndis_free_bufs(priv->npp_head);
		priv->npp_head = NULL;
		freemsg((*p)->np_m0);
	}
	priv->npp_totlen = msgdsize(m0);

	m = m0;
	len = (m->b_wptr) -(m->b_rptr);
	buf = IoAllocateMdl(m->b_rptr, len, FALSE, FALSE, NULL); 
	if (buf == NULL) {
		cmn_err(CE_CONT,"%s: Nomem!\n",__func__);
		return(ENOMEM);
	}
		
	priv->npp_head = buf;
	priv->npp_tail = buf;

	return(0);
}

int
ndis_get_supported_oids(arg, oids, oidcnt)
	void			*arg;
	ndis_oid		**oids;
	int			*oidcnt;
{
	int			len, rval, alen;
	ndis_oid		*o;

	if (arg == NULL || oids == NULL || oidcnt == NULL)
		return(EINVAL);
	len = 0;
	ndis_get_info(arg, OID_GEN_SUPPORTED_LIST, NULL, &len);

	o = kmem_zalloc(len, KM_SLEEP);

	alen = len;
	rval = ndis_get_info(arg, OID_GEN_SUPPORTED_LIST, o, &len);

	if (rval || len > alen) {
		kmem_free(o, alen);
		return(rval);
	}

	if (len != alen) {
		ndis_oid *no = kmem_zalloc(len, KM_SLEEP);

		bcopy(o, no, alen);
		kmem_free(o, alen);
		o = no;
		len = alen;
	}
	*oids = o;
	*oidcnt = len / 4;

	return(0);
}

int
ndis_set_info(arg, oid, buf, buflen)
	void			*arg;
	ndis_oid		oid;
	void			*buf;
	int			*buflen;
{
	struct ndis_softc	*sc;
	ndis_status		rval;
	ndis_handle		adapter;
	__stdcall ndis_setinfo_handler	setfunc;
	uint32_t		byteswritten = 0, bytesneeded = 0;
	int			error;
	uint8_t			irql;

	/*
	 * According to the NDIS spec, MiniportQueryInformation()
	 * and MiniportSetInformation() requests are handled serially:
	 * once one request has been issued, we must wait for it to
 	 * finish before allowing another request to proceed.
	 */

	sc = arg;
	KeAcquireSpinLock(&sc->ndis_block->nmb_lock, &irql);
	mutex_enter(&set_mtx);

	if (sc->ndis_block->nmb_pendingreq != NULL)
		return(EINVAL);
	else
		sc->ndis_block->nmb_pendingreq = (ndis_request *)sc;

	setfunc = sc->ndis_chars->nmc_setinfo_func;
	adapter = sc->ndis_block->nmb_miniportadapterctx;

	if (adapter == NULL || setfunc == NULL){
		sc->ndis_block->nmb_pendingreq = NULL;
		KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);
		mutex_exit(&set_mtx);
		return(ENXIO);
	}

	rval = MSCALL6(setfunc, adapter, oid, buf, *buflen,
	    &byteswritten, &bytesneeded);

	sc->ndis_block->nmb_pendingreq = NULL;

	KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);
	mutex_exit(&set_mtx);

	if (rval == NDIS_STATUS_PENDING) {
		cmn_err(CE_CONT,"die in set_info\n");
		mutex_enter(&ndis_req_mtx);
		cv_timedwait(&set_get_cvp, &ndis_req_mtx, ddi_get_lbolt()+5*hz);	
		mutex_exit(&ndis_req_mtx);
		rval = sc->ndis_block->nmb_setstat;
	}


	if (byteswritten)
		*buflen = byteswritten;
	if (bytesneeded)
		*buflen = bytesneeded;

	if (rval == NDIS_STATUS_INVALID_LENGTH)
		return(ENOSPC);

	if (rval == NDIS_STATUS_INVALID_OID)
		return(EINVAL);

	if (rval == NDIS_STATUS_NOT_SUPPORTED ||
	    rval == NDIS_STATUS_NOT_ACCEPTED)
		return(ENOTSUP);

	if (rval != NDIS_STATUS_SUCCESS)
		return(ENODEV);

	return(0);
}

typedef void (*ndis_senddone_func)(ndis_handle, ndis_packet *, ndis_status);

int
ndis_send_packets(arg, packets, cnt)
	void			*arg;
	ndis_packet		**packets;
	int			cnt;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_sendmulti_handler	sendfunc;
	__stdcall ndis_senddone_func		senddonefunc;
	int			i;
	ndis_packet		*p;
	uint8_t			irql;

	sc = arg;
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	if (adapter == NULL)
		return(ENXIO);
	sendfunc = sc->ndis_chars->nmc_sendmulti_func;
	senddonefunc = sc->ndis_block->nmb_senddone_func;

	NDIS_UNLOCK(sc);
	MSCALL3(sendfunc, adapter, packets, cnt);
	NDIS_LOCK(sc);

	for (i = 0; i < cnt; i++) {
		p = packets[i];
		/*
		 * Either the driver already handed the packet to
		 * ndis_txeof() due to a failure, or it wants to keep
		 * it and release it asynchronously later. Skip to the
		 * next one.
		 */
		if (p == NULL || p->np_oob.npo_status == NDIS_STATUS_PENDING){
			continue;
		}
		NDIS_UNLOCK(sc);
		MSCALL3(senddonefunc, sc->ndis_block, p, p->np_oob.npo_status);
		NDIS_LOCK(sc);
	}

	return(0);
}

int
ndis_send_packet(arg, packet)
	void			*arg;
	ndis_packet		*packet;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	ndis_status		status;
	__stdcall ndis_sendsingle_handler	sendfunc;
	__stdcall ndis_senddone_func		senddonefunc;
	uint8_t			irql;

	sc = arg;
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	if (adapter == NULL)
		return(ENXIO);
	sendfunc = sc->ndis_chars->nmc_sendsingle_func;
	senddonefunc = sc->ndis_block->nmb_senddone_func;

	if (NDIS_SERIALIZED(sc->ndis_block))
		KeAcquireSpinLock(&sc->ndis_block->nmb_lock, &irql);
	status = MSCALL3(sendfunc, adapter, packet,
	    packet->np_private.npp_flags);

	if (status == NDIS_STATUS_PENDING) {
		if (NDIS_SERIALIZED(sc->ndis_block))
			KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);
		return(0);
	}

	MSCALL3(senddonefunc, sc->ndis_block, packet, status);

	if (NDIS_SERIALIZED(sc->ndis_block))
		KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);

	return(0);
}

int
ndis_init_dma(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	int			i, error;

	return(0);
}

int
ndis_destroy_dma(arg)
	void			*arg;
{
/*
	struct ndis_softc	*sc;
	struct mbuf		*m;
	ndis_packet		*p = NULL;
	int			i;

	sc = arg;

	for (i = 0; i < sc->ndis_maxpkts; i++) {
		if (sc->ndis_txarray[i] != NULL) {
			p = sc->ndis_txarray[i];
			m = (struct mbuf *)p->np_rsvd[1];
			if (m != NULL)
				m_freem(m);
			ndis_free_packet(sc->ndis_txarray[i]);
		}
		bus_dmamap_destroy(sc->ndis_ttag, sc->ndis_tmaps[i]);
	}

	free(sc->ndis_tmaps, M_DEVBUF);

	bus_dma_tag_destroy(sc->ndis_ttag);
*/
	return(0);
}

int
ndis_reset_nic(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_reset_handler	resetfunc;
	uint8_t			addressing_reset;
	struct ifnet		*ifp;
	int			rval;
	uint8_t			irql;

	sc = arg;

	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	resetfunc = sc->ndis_chars->nmc_reset_func;

	if (adapter == NULL || resetfunc == NULL){
		INTR_UNLOCK(sc);
		return(EIO);
	}
	if (NDIS_SERIALIZED(sc->ndis_block))
		KeAcquireSpinLock(&sc->ndis_block->nmb_lock, &irql);

	rval = MSCALL2(resetfunc, &addressing_reset, adapter);

	if (NDIS_SERIALIZED(sc->ndis_block))
		KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);

	INTR_UNLOCK(sc);

	if (rval == NDIS_STATUS_PENDING) {
		mutex_enter(&ndis_req_mtx);
		cv_wait(&reset_cvp, &ndis_req_mtx);
		mutex_exit(&ndis_req_mtx);
	}

	return(0);
}

int
ndis_halt_nic(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_halt_handler	haltfunc;
	struct ifnet		*ifp;

	sc = arg;

	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	if (adapter == NULL || sc->ndis_block->nmb_interrupt == NULL) {
		INTR_UNLOCK(sc);
		return(EIO);
	}

	/*
	 * The adapter context is only valid after the init
	 * handler has been called, and is invalid once the
	 * halt handler has been called.
	 */

	haltfunc = sc->ndis_chars->nmc_halt_func;
	sc->ndis_block->nmb_miniportadapterctx = NULL;
	sc->ndis_block->nmb_interrupt = NULL;
	MSCALL1(haltfunc, adapter);
	INTR_UNLOCK(sc);

	return(0);
}

int
ndis_shutdown_nic(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_shutdown_handler	shutdownfunc;

	sc = arg;
	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	shutdownfunc = sc->ndis_chars->nmc_shutdown_handler;
	if (adapter == NULL || shutdownfunc == NULL){
		INTR_UNLOCK(sc);
		return(EIO);
	}

	if (sc->ndis_chars->nmc_rsvd0 == NULL)
		MSCALL1(shutdownfunc, adapter);
	else
		MSCALL1(shutdownfunc, sc->ndis_chars->nmc_rsvd0);

	INTR_UNLOCK(sc);
	ndis_shrink_thrqueue(8);
	TAILQ_REMOVE(&ndis_devhead, sc->ndis_block, link);
	return(0);
}

int
ndis_init_nic(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_miniport_block	*block;
        __stdcall ndis_init_handler	initfunc;
	ndis_status		status, openstatus = 0;
	ndis_medium		mediumarray[NdisMediumMax];
	uint32_t		chosenmedium, i;

	if (arg == NULL)
		return(EINVAL);

	sc = arg;
	INTR_LOCK(sc);
	block = sc->ndis_block;
	initfunc = sc->ndis_chars->nmc_init_func;

	for (i = 0; i < NdisMediumMax; i++)
		mediumarray[i] = i;

        status = MSCALL6(initfunc, &openstatus, &chosenmedium,
            mediumarray, NdisMediumMax, block, block);

	/*
	 * If the init fails, blow away the other exported routines
	 * we obtained from the driver so we can't call them later.
	 * If the init failed, none of these will work.
	 */
	if (status != NDIS_STATUS_SUCCESS) {
		sc->ndis_block->nmb_miniportadapterctx = NULL;
		INTR_UNLOCK(sc);
		return(ENXIO);
	}
	INTR_UNLOCK(sc);
	return(0);
}

void
ndis_enable_intr(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_enable_interrupts_handler	intrenbfunc;

	sc = arg;
	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	intrenbfunc = sc->ndis_chars->nmc_enable_interrupts_func;
	if (adapter == NULL || intrenbfunc == NULL){
		INTR_UNLOCK(sc);
		return;
	}
	MSCALL1(intrenbfunc, adapter);
	INTR_UNLOCK(sc);
	return;
}

void
ndis_disable_intr(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_disable_interrupts_handler	intrdisfunc;

	sc = arg;
	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	intrdisfunc = sc->ndis_chars->nmc_disable_interrupts_func;
	if (adapter == NULL || intrdisfunc == NULL){
		INTR_UNLOCK(sc);
	    return;
	}

	MSCALL1(intrdisfunc, adapter);
	INTR_UNLOCK(sc);

	return;
}

int
ndis_isr(arg, ourintr, callhandler)
	void			*arg;
	int			*ourintr;
	int			*callhandler;
{
	struct ndis_softc	*sc;
	ndis_handle		adapter;
	__stdcall ndis_isr_handler	isrfunc;
	uint8_t			accepted, queue;

	if (arg == NULL || ourintr == NULL || callhandler == NULL)
		return(EINVAL);

	sc = arg;
	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	isrfunc = sc->ndis_chars->nmc_isr_func;
	if (adapter == NULL || isrfunc == NULL){
		INTR_UNLOCK(sc);
		return(ENXIO);
	}

	MSCALL3(isrfunc, &accepted, &queue, adapter);

	*ourintr = accepted;
	*callhandler = queue;
	INTR_UNLOCK(sc);

	return(0);
}

__stdcall static void
fake_intrhand(dpc, dobj, ip, sc)
	kdpc			*dpc;
	device_object		*dobj;
	irp			*ip;
	struct ndis_softc	*sc;
{
	ndis_handle		adapter;
	ndis_interrupt_handler	intrfunc;
	uint8_t			irql;

	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	intrfunc = sc->ndis_chars->nmc_interrupt_func;
	
	if (adapter == NULL || intrfunc == NULL){
		INTR_UNLOCK(sc);
		return;
	}

	MSCALL1(intrfunc, adapter);
	/* If there's a MiniportEnableInterrupt() routine, call it. */
	INTR_UNLOCK(sc);

	ndis_enable_intr(sc);

	return;
}

void ndis_intrhandler(void *arg)
{
	ndis_handle		adapter;
	ndis_interrupt_handler	intrfunc;
	uint8_t			irql;
	struct ndis_softc	*sc = (struct ndis_softc *)arg;

	INTR_LOCK(sc);
	adapter = sc->ndis_block->nmb_miniportadapterctx;
	intrfunc = sc->ndis_chars->nmc_interrupt_func;
	
	if (adapter == NULL || intrfunc == NULL){
		INTR_UNLOCK(sc);
		return;
	}

	MSCALL1(intrfunc, adapter);
	/* If there's a MiniportEnableInterrupt() routine, call it. */

	INTR_UNLOCK(sc);
	ndis_enable_intr(sc);

	return;
}

int
ndis_get_info(arg, oid, buf, buflen)
	void			*arg;
	ndis_oid		oid;
	void			*buf;
	int			*buflen;
{
	struct ndis_softc	*sc;
	ndis_status		rval;
	ndis_handle		adapter;
	__stdcall ndis_queryinfo_handler	queryfunc;
	uint32_t		byteswritten = 0, bytesneeded = 0;
	int			error;
	uint8_t			irql;

	sc = arg;
	KeAcquireSpinLock(&sc->ndis_block->nmb_lock, &irql);
	mutex_enter(&get_mtx);

	if (sc->ndis_block->nmb_pendingreq != NULL)
		return(EINVAL);
	else
		sc->ndis_block->nmb_pendingreq = (ndis_request *)sc;

	queryfunc = sc->ndis_chars->nmc_queryinfo_func;
	adapter = sc->ndis_block->nmb_miniportadapterctx;

	if (adapter == NULL || queryfunc == NULL){
		sc->ndis_block->nmb_pendingreq = NULL;
		KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);
		mutex_exit(&get_mtx);
		return(ENXIO);
	}

	rval = MSCALL6(queryfunc, adapter, oid, buf, *buflen,
	    &byteswritten, &bytesneeded);

	sc->ndis_block->nmb_pendingreq = NULL;

	KeReleaseSpinLock(&sc->ndis_block->nmb_lock, irql);
	mutex_exit(&get_mtx);
	/* Wait for requests that block. */

	if (rval == NDIS_STATUS_PENDING) {
		cmn_err(CE_CONT,"die in get_info\n");
		mutex_enter(&ndis_req_mtx);
		cv_timedwait(&set_get_cvp, &ndis_req_mtx, ddi_get_lbolt()+5*hz);
		mutex_exit(&ndis_req_mtx);
		rval = sc->ndis_block->nmb_getstat;
	}

	if (byteswritten)
		*buflen = byteswritten;
	if (bytesneeded)
		*buflen = bytesneeded;

	if (rval == NDIS_STATUS_INVALID_LENGTH ||
	    rval == NDIS_STATUS_BUFFER_TOO_SHORT)
		return(ENOSPC);

	if (rval == NDIS_STATUS_INVALID_OID)
		return(EINVAL);

	if (rval == NDIS_STATUS_NOT_SUPPORTED ||
	    rval == NDIS_STATUS_NOT_ACCEPTED)
		return(ENOTSUP);

	if (rval != NDIS_STATUS_SUCCESS)
		return(ENODEV);

	return(0);
}

uint32_t instence_nr=0;
struct ndis_softc *curSC=NULL;

__stdcall uint32_t
NdisAddDevice(drv, pdo,sc)
	driver_object		*drv;
	device_object		*pdo;
	struct ndis_softc	*sc;
{
	device_object		*fdo;
	ndis_miniport_block	*block;
	uint32_t		status;

	status = IoCreateDevice(drv, sizeof(ndis_miniport_block), NULL,
	    FILE_DEVICE_UNKNOWN, 0, FALSE, &fdo);

	if (status != STATUS_SUCCESS)
		return(status);

	block = fdo->do_devext;
	block->nmb_deviceobj = fdo;
	block->nmb_physdeviceobj = pdo;
	block->nmb_nextdeviceobj = IoAttachDeviceToDeviceStack(fdo, pdo);
	KeInitializeSpinLock(&block->nmb_lock);

	/*
	 * Stash pointers to the miniport block and miniport
	 * characteristics info in the if_ndis softc so the
	 * UNIX wrapper driver can get to them later.
         */
	sc->ndis_block = block;
	sc->ndis_chars = IoGetDriverObjectExtension(drv, (void *)1);

	IoInitializeDpcRequest(fdo, fake_intrhand);

	/* Finish up BSD-specific setup. */

	block->nmb_signature = (void *)0xcafebabe;
	block->nmb_status_func = kernndis_functbl[0].ipt_wrap;
	block->nmb_statusdone_func = kernndis_functbl[1].ipt_wrap;
	block->nmb_setdone_func = kernndis_functbl[2].ipt_wrap;
	block->nmb_querydone_func = kernndis_functbl[3].ipt_wrap;
	block->nmb_resetdone_func = kernndis_functbl[4].ipt_wrap;
	block->nmb_sendrsrc_func = kernndis_functbl[5].ipt_wrap;
	block->nmb_pendingreq = NULL;

	block->nmb_ifp = sc;
	block->nmb_devobj.do_rsvd = block;
	block->devinfo = sc->devinfo_this;
	curSC = sc;

	ndis_enlarge_thrqueue(8);

	TAILQ_INSERT_TAIL(&ndis_devhead, block, link);
	instence_nr++;
	return (STATUS_SUCCESS);
}

int
ndis_unload_driver(arg)
	void			*arg;
{
	struct ndis_softc	*sc;
	device_object		*fdo;

	sc = arg;

	kmem_free(sc->ndis_block->nmb_rlist, 
		sc->ndis_block->nmb_rlist_size);

	ndis_flush_sysctls(sc);

	ndis_shrink_thrqueue(8);
	TAILQ_REMOVE(&ndis_devhead, sc->ndis_block, link);

	fdo = sc->ndis_block->nmb_deviceobj;
	IoDetachDevice(sc->ndis_block->nmb_nextdeviceobj);
	IoDeleteDevice(fdo);
	instence_nr--;
	return(0);
}

static struct modlmisc modlmisc = {
	&mod_miscops, "NDIS API 1.2.1"
};

static struct modlinkage modlinkage = {
	MODREV_1, (void *)&modlmisc, NULL
};

int _init(void)
{
	/* Initialize subsystems */
	int error;
	image_patch_table	*patch;

	error = mod_install(&modlinkage);

	windrv_libinit();
	hal_libinit();
	ndis_libinit();
	ntoskrnl_libinit();

		patch = kernndis_functbl;
		while (patch->ipt_func != NULL) {
			windrv_wrap((funcptr)patch->ipt_func,
			    (funcptr *)&patch->ipt_wrap);
			patch++;
		}

		/* Initialize TX buffer UMA zone. */

		ndis_create_kthreads();
		TAILQ_INIT(&ndis_devhead);
	return (error);
}

int _fini(void)
{
	int error;
	image_patch_table	*patch;
	/* stop kthreads */
	if(instence_nr)
		return(-1);
	error = mod_remove(&modlinkage);

	ndis_destroy_kthreads();
	/* Shut down subsystems */
	if (TAILQ_FIRST(&ndis_devhead) == NULL) {
		hal_libfini();
		ndis_libfini();
		ntoskrnl_libfini();
		windrv_libfini();

		patch = kernndis_functbl;
		while (patch->ipt_func != NULL) {
			windrv_unwrap(patch->ipt_wrap);
			patch++;
		}
	}
	return (error);
}

int _info(struct modinfo *modinfop)
{
	return (mod_info(&modlinkage, modinfop));
}
