
/*
 * Copyright 2008 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * Copyright (c) 2003
 *	Bill Paul <wpaul@windriver.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/dditypes.h>
#include <sys/stream.h>
#include <sys/termio.h>
#include <sys/thread.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/cmn_err.h>
#include <sys/stropts.h>
#include <sys/strtty.h>
#include <sys/kbio.h>
#include <sys/cred.h>
#include <sys/stat.h>
#include <sys/consdev.h>
#include <sys/kmem.h>
#include <sys/modctl.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>
#include <sys/pci.h>
#include <sys/errno.h>
#include <sys/gld.h>
#include <sys/dlpi.h>
#include <dls.h>
#include <sys/ethernet.h>
#include <if_ieee80211.h>

#include <inet/common.h>
#include <inet/nd.h>
#include <inet/mi.h>
#include <sys/strsun.h>

#include <queue.h>

#include <pe_var.h>
#include <resource_var.h>
#include <ntoskrnl_var.h>
#include <ndis_var.h>
#include <hal_var.h>
#include <cfg_var.h>
#include <if_ndisvar.h>
#include <wifi_ioctl.h>

char _depends_on[] = "misc/ndisapi misc/mac ip";

#define NDIS_IMAGE
#define NDIS_REGVALS

#include "ndis.h"

struct ndis_pci_type ndis_devs[] = {
#ifdef NDIS_PCI_DEV_TABLE
	NDIS_PCI_DEV_TABLE
#endif
	{ 0, 0, 0, NULL }
};

unsigned char *drv_data_inuse;
uint32_t rxcnt;
uint32_t uploadcnt;

int ndis_attach(dev_info_t *, ddi_attach_cmd_t);
static int ndis_detach(dev_info_t *, ddi_detach_cmd_t);
static int ndis_quiesce(dev_info_t *);

__stdcall static void ndis_txeof(ndis_handle, ndis_packet *, ndis_status);
__stdcall static void ndis_rxeof(ndis_handle, ndis_packet **, uint32_t);
	
__stdcall static void ndis_linksts(ndis_handle, ndis_status, void *, uint32_t);
__stdcall static void ndis_linksts_done	(ndis_handle);
void iotest_ping(struct ndis_softc *sc);

static funcptr ndis_txeof_wrap;
static funcptr ndis_rxeof_wrap;
static funcptr ndis_linksts_wrap;
static funcptr ndis_linksts_done_wrap;

static uint32_t ndis_intr(caddr_t arg);
static void ndis_tick		(void *);
static void ndis_ticktask	(void *);

static mblk_t *ndis_m_tx(void *arg, mblk_t *mp);
static int32_t ndis_m_start(void *);
static void ndis_m_stop  (void *);

static int ndis_tx_begin(struct ndis_softc *sc, mblk_t *mp);

static int  ndis_get_assoc      (struct ndis_softc *, ndis_wlan_bssid_ex **);
static void ndis_getstate_80211	(struct ndis_softc *);
static void ndis_setstate_80211	(struct ndis_softc *);

uint32_t ieee80211_ieee2mhz(uint32_t chan, uint32_t  flags);
uint32_t ieee80211_mhz2ieee(uint32_t freq, uint32_t flags);
uint32_t ieee80211_chan2ieee(ieee80211com_t *isc, struct ieee80211channel *ch);

#define NDISINF(x) \
        if (ndis_debug & NDISINFMSG ) cmn_err x

#define NDISDBG(x) \
        if (ndis_debug & NDISDBGMSG ) cmn_err x

#define NWAMDBG(x) \
        if (ndis_debug & NDSINWAMSG ) cmn_err x

#define NDIS11NDBG(x) \
	if (ndis_debug & NDIS80211N ) cmn_err x

#define NDISIOCTL(x) \
	if (ndis_debug & NDISDIOCTL ) cmn_err x

#define NDISRATES(x) \
	if (ndis_debug & NDISDRATES ) cmn_err x

#define NDISCFGPHY(x) \
	if (ndis_debug & NDISDCFGPHY ) cmn_err x

#define NDIS_DRIVER_NAME 	"bcmndis"

static int ndisdrv_loaded = 0;
static int tickstop;
void *ndis_soft_state_p = NULL;

uint32_t identify;
DDI_DEFINE_STREAM_OPS(ndis_dev_ops, nulldev, nulldev, ndis_attach, ndis_detach,
    nodev, NULL, D_MP, NULL, ndis_quiesce);

static struct modldrv ndis_modldrv = {
	&mod_driverops,		/* Type of module.  This one is a driver */
	"Ndiswrapper 1.2.1",	/* short description */
	&ndis_dev_ops		/* driver specific ops */
};

static struct modlinkage modlinkage = {
	MODREV_1, (void *)&ndis_modldrv, NULL
};

wifi_data_t g_wd;
uint8_t wifi_bcastaddr[]	= { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
static uint8_t wifi_ietfmagic[]	= { 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };
static uint8_t wifi_ieeemagic[]	= { 0xaa, 0xaa, 0x03, 0x00, 0x00, 0xf8 };
extern int *ndis_i_pcnt;

#define	IEEE80211_IS_MULTICAST(addr)	\
	((((uint8_t *)addr)[0]) & 0x01)

static int
__mac_wifi_unicst_verify(const void *addr, void *pdata)
{
	/* If it's not a group address, then it's a valid unicast address. */
	return (IEEE80211_IS_MULTICAST(addr) ? EINVAL : 0);
}

/* ARGSUSED */
static int
__mac_wifi_multicst_verify(const void *addr, void *pdata)
{
	/* The address must be a group address. */
	if (!IEEE80211_IS_MULTICAST(addr))
		return (EINVAL);
	/* The address must not be the media broadcast address. */
	if (bcmp(addr, wifi_bcastaddr, sizeof (wifi_bcastaddr)) == 0)
		return (EINVAL);
	return (0);
}

static boolean_t
__mac_wifi_sap_verify(uint32_t sap, uint32_t *bind_sap, void *pdata)
{
	if (sap >= ETHERTYPE_802_MIN && sap <= ETHERTYPE_MAX) {
		if (bind_sap != NULL)
			*bind_sap = sap;
		return (B_TRUE);
	}

	if (sap <= ETHERMTU) {
		if (bind_sap != NULL)
			*bind_sap = DLS_SAP_LLC;
		return (B_TRUE);
	}
	return (B_FALSE);
}


static mblk_t *
mac_wifi_header_new(const void *saddr, const void *daddr, uint32_t sap,
    void *pdata, mblk_t *payload, size_t extra_len)
{
	struct ieee80211_frame	*wh;
	struct ieee80211_llc	*llc;
	mblk_t			*mp;
	wifi_data_t		*wdp = pdata;

	if (!__mac_wifi_sap_verify(sap, NULL, NULL))
		return (NULL);

	if ((mp = allocb(WIFI_HDRSIZE + extra_len, BPRI_HI)) == NULL)
		return (NULL);
	bzero(mp->b_rptr, WIFI_HDRSIZE + extra_len);

	/*
	 * Fill in the fixed parts of the ieee80211_frame.
	 */
	wh = (struct ieee80211_frame *)mp->b_rptr;
	mp->b_wptr += sizeof (struct ieee80211_frame);
	wh->i_fc[0] = IEEE80211_FC0_VERSION_0 | IEEE80211_FC0_TYPE_DATA;

	switch (wdp->wd_opmode) {
	case IEEE80211_M_STA:
		wh->i_fc[1] = IEEE80211_FC1_DIR_TODS;
		IEEE80211_ADDR_COPY(wh->i_addr1, wdp->wd_bssid);
		IEEE80211_ADDR_COPY(wh->i_addr2, saddr);
		IEEE80211_ADDR_COPY(wh->i_addr3, daddr);
		break;

	case IEEE80211_M_IBSS:
	case IEEE80211_M_AHDEMO:
		wh->i_fc[1] = IEEE80211_FC1_DIR_NODS;
		IEEE80211_ADDR_COPY(wh->i_addr1, daddr);
		IEEE80211_ADDR_COPY(wh->i_addr2, saddr);
		IEEE80211_ADDR_COPY(wh->i_addr3, wdp->wd_bssid);
		break;

	case IEEE80211_M_HOSTAP:
		wh->i_fc[1] = IEEE80211_FC1_DIR_FROMDS;
		IEEE80211_ADDR_COPY(wh->i_addr1, daddr);
		IEEE80211_ADDR_COPY(wh->i_addr2, wdp->wd_bssid);
		IEEE80211_ADDR_COPY(wh->i_addr3, saddr);
		break;
	}

	/*
	 * Fill in the fixed parts of the WEP-portion of the frame.
	 */
	if (wdp->wd_secalloc == WIFI_SEC_WEP) {
		wh->i_fc[1] |= IEEE80211_FC1_WEP;
		/*
		 * The actual contents of the WEP-portion of the packet
		 * are computed when the packet is sent -- for now, we
		 * just need to account for the size.
		 */
		mp->b_wptr += IEEE80211_WEP_IVLEN + IEEE80211_WEP_KIDLEN;
	}

	/*
	 * Fill in the fixed parts of the ieee80211_llc header.
	 */
	llc = (struct ieee80211_llc *)mp->b_wptr;
	mp->b_wptr += sizeof (struct ieee80211_llc);
	bcopy(wifi_ietfmagic, llc, sizeof (wifi_ietfmagic));
	llc->illc_ether_type = htons(sap);

	return (mp);
}


static mblk_t *
mac_wifi_header_fake(mblk_t *mp, void *pdata)
{
	struct ether_header	*ehp;
	mblk_t			*llmp;

	if (MBLKL(mp) < sizeof (struct ether_header))
		return (NULL);

	ehp = (struct ether_header *)mp->b_rptr;
	llmp = mac_wifi_header_new(&ehp->ether_shost, &ehp->ether_dhost,
	    ntohs(ehp->ether_type), pdata, NULL, 0);
	if (llmp == NULL)
		return (NULL);

	/*
	 * The plugin framework guarantees that we have the only reference
	 * to the mblk_t, so we can safely modify it.
	 */
	mp->b_rptr += sizeof (struct ether_header);
	llmp->b_cont = mp;
	return (llmp);
}

static int
mac_wifi_header_check(mblk_t *mp, void *pdata, mac_header_info_t *mhp)
{
	struct ieee80211_frame	*wh;
	struct ieee80211_llc	*llc;
	uchar_t			*llcp;

	if (MBLKL(mp) < sizeof (struct ieee80211_frame))
		return (EINVAL);

	wh = (struct ieee80211_frame *)mp->b_rptr;
	llcp = mp->b_rptr + sizeof (struct ieee80211_frame);

	/*
	 * When we receive frames from other hosts, the hardware will have
	 * already performed WEP decryption, and thus there will not be a WEP
	 * portion.  However, when we receive a loopback copy of our own
	 * packets, it will still have a WEP portion.  Skip past it to get to
	 * the LLC header.
	 */
	if (wh->i_fc[1] & IEEE80211_FC1_WEP)
		llcp += IEEE80211_WEP_IVLEN + IEEE80211_WEP_KIDLEN;

	if (mp->b_wptr - llcp < sizeof (struct ieee80211_llc))
		return (EINVAL);

	llc = (struct ieee80211_llc *)llcp;
	mhp->mhi_origsap = ntohs(llc->illc_ether_type);
	mhp->mhi_bindsap = mhp->mhi_origsap;
	mhp->mhi_pktsize = 0;
	mhp->mhi_hdrsize = llcp + sizeof (*llc) - mp->b_rptr;

	/*
	 * Verify the LLC header is one of the known formats.  As per MSFT's
	 * convention, if the header is using IEEE 802.1H encapsulation, then
	 * treat the LLC header as data.  As per DL_ETHER custom when treating
	 * the LLC header as data, set the mhi_bindsap to be DLS_SAP_LLC, and
	 * assume mhi_origsap contains the data length.
	 */
	if (bcmp(llc, wifi_ieeemagic, sizeof (wifi_ieeemagic)) == 0) {
		mhp->mhi_bindsap = DLS_SAP_LLC;
		mhp->mhi_hdrsize -= sizeof (*llc);
		mhp->mhi_pktsize = mhp->mhi_hdrsize + mhp->mhi_origsap;
	} else if (bcmp(llc, wifi_ietfmagic, sizeof (wifi_ietfmagic)) != 0) {
		return (EINVAL);
	}

	switch (wh->i_fc[1] & IEEE80211_FC1_DIR_MASK) {
	case IEEE80211_FC1_DIR_NODS:
		mhp->mhi_daddr = wh->i_addr1;
		mhp->mhi_saddr = wh->i_addr2;
		break;

	case IEEE80211_FC1_DIR_TODS:
		mhp->mhi_daddr = wh->i_addr3;
		mhp->mhi_saddr = wh->i_addr2;
		break;

	case IEEE80211_FC1_DIR_FROMDS:
		mhp->mhi_daddr = wh->i_addr1;
		mhp->mhi_saddr = wh->i_addr3;
		break;

	case IEEE80211_FC1_DIR_DSTODS:
		/* We don't support AP-to-AP mode yet */
		return (ENOTSUP);
	}

	if (__mac_wifi_unicst_verify(mhp->mhi_daddr, NULL) == 0)
		mhp->mhi_dsttype = MAC_ADDRTYPE_UNICAST;
	else if (__mac_wifi_multicst_verify(mhp->mhi_daddr, NULL) == 0)
		mhp->mhi_dsttype = MAC_ADDRTYPE_MULTICAST;
	else
		mhp->mhi_dsttype = MAC_ADDRTYPE_BROADCAST;

	return (0);
}

static mblk_t *
mac_wifi_header_backout(mblk_t *mp, void *pdata)
{
	mac_header_info_t	mhi;
	struct ether_header	eh;

	if (mac_wifi_header_check(mp, pdata, &mhi) != 0) {
		/*
		 * The plugin framework guarantees the header is properly
		 * formed, so this should never happen.
		 */
		return (NULL);
	}

	/*
	 * The plugin framework guarantees that we have the only reference to
	 * the mblk_t and the underlying dblk_t, so we can safely modify it.
	 */

	IEEE80211_ADDR_COPY(&eh.ether_dhost, mhi.mhi_daddr);
	IEEE80211_ADDR_COPY(&eh.ether_shost, mhi.mhi_saddr);
	eh.ether_type = htons(mhi.mhi_origsap);

	mp->b_rptr += mhi.mhi_hdrsize - sizeof (struct ether_header);
	bcopy(&eh, mp->b_rptr, sizeof (struct ether_header));
	return (mp);
}

int _init(void)
{
	int e;
	if (e = ddi_soft_state_init(&ndis_soft_state_p, 
	    sizeof (struct ndis_softc), 1))
		return (e);
	
	mac_init_ops(&ndis_dev_ops, NDIS_DRIVER_NAME);
	if (e = mod_install(&modlinkage)){
		ddi_soft_state_fini(&ndis_soft_state_p);
		mac_fini_ops(&ndis_dev_ops);
	}
	return (e);
}

int _fini(void)
{
	int e;
	if (e = mod_remove(&modlinkage))
		return (e);
	mac_fini_ops(&ndis_dev_ops);
	ddi_soft_state_fini(&ndis_soft_state_p);
	return (e);
}

int _info(struct modinfo *modinfop)
{
	return (mod_info(&modlinkage, modinfop));
}

static int32_t
ndis_cfg_essid(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	ndis_80211_ssid ssid;
	wldp_t *infp;
	char *value;
	uint16_t i;
	wldp_t *outfp;
	char *buf;
	int rval;
	int len;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,"ieee80211: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		char ssid[36];
		if (mi_strlen((const char *)isc->isc_des_essid) == 0) {
			outfp->wldp_length = WIFI_BUF_OFFSET +
			    offsetof(wl_essid_t, wl_essid_essid) +
			    mi_strlen((const char *)
			    isc->isc_bss->ni_essid);
			((wl_essid_t *)(outfp->wldp_buf))->wl_essid_length =
			    mi_strlen((const char *)
			    isc->isc_bss->ni_essid);
			bcopy(isc->isc_bss->ni_essid,
			    buf + WIFI_BUF_OFFSET +
			    offsetof(wl_essid_t, wl_essid_essid),
			    mi_strlen((const char *)
			    isc->isc_bss->ni_essid));
		} else {
			outfp->wldp_length = WIFI_BUF_OFFSET +
			    offsetof(wl_essid_t, wl_essid_essid) +
			    mi_strlen((const char *)isc->isc_des_essid);
			((wl_essid_t *)(outfp->wldp_buf))->wl_essid_length =
			    mi_strlen((const char *)isc->isc_des_essid);
			bcopy(isc->isc_des_essid,
			    buf + WIFI_BUF_OFFSET +
			    offsetof(wl_essid_t, wl_essid_essid),
			    mi_strlen((const char *)
			    isc->isc_des_essid));
		}
		outfp->wldp_result = WL_SUCCESS;
		
	} else if (cmd == WLAN_SET_PARAM) {
		value = ((wl_essid_t *)(infp->wldp_buf))->wl_essid_essid;
		bzero(isc->isc_des_essid, IEEE80211_NWID_LEN);
		if (mi_strlen(value) == 0) {
			bzero(isc->isc_des_essid, IEEE80211_NWID_LEN);
			isc->isc_des_esslen = 0;
		} else {
			len = mi_strlen((const char *)value);
			if (len > IEEE80211_NWID_LEN)
				len = IEEE80211_NWID_LEN;
			bcopy(value, isc->isc_des_essid, len);
			isc->isc_des_esslen = len;
			len = sizeof(ndis_80211_ssid);
			bzero((char *)&ssid, len);
			ssid.ns_ssidlen = isc->isc_des_esslen;
			if (ssid.ns_ssidlen == 0)
				ssid.ns_ssidlen = 1;
			else
				bcopy(isc->isc_des_essid, ssid.ns_ssid, ssid.ns_ssidlen);
			NDISDBG((CE_CONT,"%s: bssid=%s\n",__func__, isc->isc_des_essid));

			outfp->wldp_length = WIFI_BUF_OFFSET;

			rval = ndis_set_info(sc, OID_802_11_SSID, (void *)&ssid, &len);
			if (rval){
				NDISDBG((CE_CONT,"set ssid failed: %d\n", rval));
				outfp->wldp_result = WL_HW_ERROR;
				goto done;
			}
			outfp->wldp_result = WL_SUCCESS;
		}		
	}

done:
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
}

static int32_t
ndis_cfg_bssid(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char bssid[6];
	uint16_t i;
	int rval;
	wldp_t *outfp;
	char *buf;
	int len;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT, "ndis: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN));
		return (ENOMEM);
	}
	
	outfp = (wldp_t *)buf;
	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		len = sizeof(bssid);
		rval = ndis_get_info(sc, OID_802_11_BSSID, (void *)bssid, &len);
		if(rval) {
			NDISDBG((CE_CONT,
			    "get bssid failed: %d\n", rval));	
			bzero(buf + WIFI_BUF_OFFSET, sizeof(wl_bssid_t));
			outfp->wldp_result = WL_HW_ERROR;
		}else{
			bcopy(bssid, buf + WIFI_BUF_OFFSET, sizeof(wl_bssid_t));
			outfp->wldp_result = WL_SUCCESS;
		}		
		outfp->wldp_length =  WIFI_BUF_OFFSET + sizeof(wl_bssid_t);
	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_READONLY;	
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}


static int32_t
ndis_cfg_encryption(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char *value;
	uint16_t i;
	wldp_t *outfp;
	char *buf;
	uint32_t arg,len=0;
	int rval=0;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		printf("ndis: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN);
		return (ENOMEM);
	}
	
	outfp = (wldp_t *)buf;
	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));


	outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_encryption_t);
	    
	if (cmd == WLAN_GET_PARAM) {
		*(wl_encryption_t *)(outfp->wldp_buf) =
		    (isc->isc_flags & IEEE80211_F_WEPON) ? 1 : 0;
		outfp->wldp_result = WL_SUCCESS;
	} else if (cmd == WLAN_SET_PARAM) {
		if (*(wl_encryption_t *)(infp->wldp_buf) ==
		    WL_NOENCRYPTION) {
			isc->isc_flags &= ~IEEE80211_F_WEPON;
		} else if (*(wl_encryption_t *)(infp->wldp_buf) ==
		    WL_ENC_WEP) {
			isc->isc_flags |= IEEE80211_F_WEPON;
		}
		
		if (isc->isc_flags & IEEE80211_F_WEPON) {
			arg = NDIS_80211_WEPSTAT_ENABLED;
			len = sizeof(arg);
			rval = ndis_set_info(sc, OID_802_11_WEP_STATUS, (void *)&arg, &len);
			if (rval){
				NDISDBG((CE_CONT, "enable WEP failed: %d\n", rval));
				outfp->wldp_result = WL_HW_ERROR;
				goto done;
			}
/*			arg = NDIS_80211_PRIVFILT_8021XWEP;
			len = sizeof(arg);
			rval = ndis_set_info(sc, OID_802_11_PRIVACY_FILTER, (void *)&arg, &len);			
			if (rval){
				NDISDBG((CE_CONT, "set privacy filter failed: %d\n", rval));
				outfp->wldp_result = WL_HW_ERROR;
				goto done;
			}
*/
		} else {
			arg = NDIS_80211_WEPSTAT_DISABLED;
			len = sizeof(arg);
			rval = ndis_set_info(sc, OID_802_11_WEP_STATUS, (void *)&arg, &len);
			if (rval){
				NDISDBG((CE_CONT, "disable WEP failed: %d\n", rval));
				outfp->wldp_result = WL_HW_ERROR;
				goto done;
			}			
		}
		
		NDISDBG((CE_CONT,"ndis_getset: set encryption=%d\n",
		    (isc->isc_flags & IEEE80211_F_WEPON) ? 1 : 0));

		outfp->wldp_result = WL_SUCCESS;
	}

done:
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
}


static int32_t
ndis_cfg_wepkeyindex(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char *value;
	uint16_t i;
	wldp_t *outfp;
	char *buf;
	int  len, rval = 0;
	uint32_t		arg;
	ndis_80211_wep	wep;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,"ndis: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_wep_key_id_t);

	if (cmd == WLAN_GET_PARAM) {
		*(wl_wep_key_id_t *)(outfp->wldp_buf) =
		    isc->isc_wep_txkey;
		outfp->wldp_result = WL_SUCCESS;
	} else if (cmd == WLAN_SET_PARAM) {
		isc->isc_wep_txkey = *(wl_wep_key_id_t *)(infp->wldp_buf);
		NDISDBG((CE_CONT,"set wepkeyid=%d\n",
		    isc->isc_wep_txkey));
		for (i = 0; i < IEEE80211_WEP_NKID; i++) {
			if (isc->isc_nw_keys[i].wk_len) {
				bzero((char *)&wep, sizeof(wep));
				wep.nw_keylen = isc->isc_nw_keys[i].wk_len;
				
				wep.nw_keyidx = i;
				wep.nw_length = (sizeof(uint32_t) * 3)
				    + wep.nw_keylen;
				if (i == (isc->isc_wep_txkey))
					wep.nw_keyidx |= NDIS_80211_WEPKEY_TX;
				else
					wep.nw_keyidx &= ~NDIS_80211_WEPKEY_TX;

				bcopy(isc->isc_nw_keys[i].wk_key,
				    wep.nw_keydata, wep.nw_length);
				len = sizeof(wep);
				rval = ndis_set_info(sc,
				    OID_802_11_ADD_WEP, &wep, &len);	
				if (rval){
					NDISDBG((CE_CONT,
					    "set wepkey failed: %d\n", rval));
					outfp->wldp_result = WL_HW_ERROR;
					goto done;
				}
			}
		}
		outfp->wldp_result = WL_SUCCESS;
	}

done:
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
}


static int32_t
ndis_cfg_setwepkey(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char *value;
	uint16_t i;
	wldp_t *outfp;
	char *buf;
	int  len, rval = 0;
	uint32_t		arg;
	ndis_80211_wep	wep;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,"ndis: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));
	
	outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_wep_key_tab_t);

	if (cmd == WLAN_GET_PARAM) {
		outfp->wldp_result = WL_WRITEONLY;
	} else if (cmd == WLAN_SET_PARAM) {
		wl_wep_key_t *p_wepkey_tab;
		p_wepkey_tab = (wl_wep_key_t *)(infp->wldp_buf);

		for (i = 0; i < 4; i++) {
			if (p_wepkey_tab[i].wl_wep_operation == WL_ADD) {
				isc->isc_nw_keys[i].wk_len =
				    p_wepkey_tab[i].wl_wep_length;
				NDISDBG((CE_CONT,"isc->isc_nw_keys[%d].wk_len=%d"
					,i,p_wepkey_tab[i].wl_wep_length));
				bcopy(p_wepkey_tab[i].wl_wep_key,
				    isc->isc_nw_keys[i].wk_key,
				    p_wepkey_tab[i].wl_wep_length);
			}
		}

		isc->isc_flags |=IEEE80211_F_WEPON; 

		for (i = 0; i < IEEE80211_WEP_NKID; i++) {
			if (isc->isc_nw_keys[i].wk_len) {
				bzero((char *)&wep, sizeof(wep));
				wep.nw_keylen = isc->isc_nw_keys[i].wk_len;
				
				wep.nw_keyidx = i;
				wep.nw_length = (sizeof(uint32_t) * 3)
				    + wep.nw_keylen;
				if (i == (isc->isc_wep_txkey))
					wep.nw_keyidx |= NDIS_80211_WEPKEY_TX;
				else
					wep.nw_keyidx &= ~NDIS_80211_WEPKEY_TX;

				bcopy(isc->isc_nw_keys[i].wk_key,
				    wep.nw_keydata, wep.nw_length);
				len = sizeof(wep);
				rval = ndis_set_info(sc,
				    OID_802_11_ADD_WEP, &wep, &len);	
				if (rval){
					NDISDBG((CE_CONT,
					    "set wepkey failed: %d\n", rval));
					outfp->wldp_result = WL_HW_ERROR;
					goto done;
				}
			}
		}
		outfp->wldp_result = WL_SUCCESS;
	}
done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
}

static int32_t
ndis_cfg_scan(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	wldp_t *outfp;
	char *buf;
	uint32_t len, rval,i;
	
	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			  "ndis: can not alloc so much memory!(%d)\n",
			  MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;
	bcopy(mp->b_rptr, buf,  sizeof(wldp_t));

	NDIS_UNLOCK(sc);
	ndis_m_start(sc);
	NDIS_LOCK(sc);

	NDISDBG((CE_CONT,"scan command issued...\n"));
	outfp->wldp_result = WL_SUCCESS;

	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;
}

#define	IEEE80211_MAX_WPA_IE		40	/* IEEE802.11i */
#define GENERIC_INFO_ELEM	0xdd
#define RSN_INFO_ELEM	0x30

static void wpa_ie_parser(wl_ess_conf_t *ap, uint8_t *ie,
 				    size_t ie_len)
{
	uint8_t *pos = ie;
	uint8_t *end = ie + ie_len;
 
	if (ie_len < sizeof(struct ndis_80211_fixed_ies))
		return;

		pos += sizeof(struct ndis_80211_fixed_ies);
	while (pos + 1 < end && pos + 2 + pos[1] <= end) {
	 		uint8_t ielen = 2 + pos[1];
	 		if (ielen > IEEE80211_MAX_WPA_IE) {
				pos += ielen;
				continue;
			}
			if (pos[0] == GENERIC_INFO_ELEM && pos[1] >= 4 &&
			    memcmp(pos + 2, "\x00\x50\xf2\x01", 4) == 0) {
				ap->wl_ess_conf_reserved[0] = 1;
			} else if (pos[0] == RSN_INFO_ELEM) {
				ap->wl_ess_conf_reserved[0] = 1;
 			}
			pos += ielen;
	}
}

static int
ndis_nettype_parser(wl_ess_conf_t *ap, ndis_wlan_bssid_ex *wb)
{
	if(ap == NULL || wb ==NULL)
		return -1;
	
	switch(wb->nwbx_nettype){
	case NDIS_80211_NETTYPE_11FH:
		NDISDBG((CE_CONT,"NDIS_80211_NETTYPE_11FH\n"));
		wl_fhss_t *fhss = (wl_fhss_t *)&((ap->wl_phy_conf).wl_phy_fhss_conf);
		fhss->wl_fhss_subtype = WL_FHSS;
		fhss->wl_fhss_dwelltime = wb->nwbx_config.nc_fhconfig.ncf_dwelltime;
		fhss->wl_fhss_channel = ieee80211_mhz2ieee(wb->nwbx_config.nc_dsconfig / 1000, 0);
		break;
	case NDIS_80211_NETTYPE_11DS:
		NDISDBG((CE_CONT,"NDIS_80211_NETTYPE_11DS\n"));
		wl_dsss_t *dsss = (wl_dsss_t *)&((ap->wl_phy_conf).wl_phy_dsss_conf);
		dsss->wl_dsss_subtype = WL_DSSS;
		dsss->wl_dsss_channel = ieee80211_mhz2ieee(wb->nwbx_config.nc_dsconfig / 1000, 0);
		break;
	case NDIS_80211_NETTYPE_11OFDM5:
		NDISDBG((CE_CONT,"NDIS_80211_NETTYPE_11OFDM5\n"));
		wl_ofdm_t *ofdm = (wl_ofdm_t *)&((ap->wl_phy_conf).wl_phy_ofdm_conf);
		ofdm->wl_ofdm_subtype = WL_OFDM;		
		ofdm->wl_ofdm_frequency = wb->nwbx_config.nc_dsconfig / 1000;
		break;				
	case NDIS_80211_NETTYPE_11OFDM24:
		NDISDBG((CE_CONT,"NDIS_80211_NETTYPE_11OFDM24\n"));
		wl_erp_t *erp = (wl_erp_t *)&((ap->wl_phy_conf).wl_phy_erp_conf);
		erp->wl_erp_subtype = WL_ERP;		
		erp->wl_erp_channel = ieee80211_mhz2ieee(wb->nwbx_config.nc_dsconfig / 1000, 0);
		break;
	default:
		cmn_err(CE_CONT,"unknown nettype 0x%08x\n", wb->nwbx_nettype);
		return (-1);
	}
	return (0);
}

static int32_t
ndis_cfg_esslist(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char *value;
	uint16_t i,j,k;
	wldp_t *outfp;
	char *buf,*db;
	uint32_t len, rval;
	ndis_80211_bssid_list_ex *ess_list;
	ndis_wlan_bssid_ex *wb;
	uint32_t rssi;
	
	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			  "ieee80211: can not alloc so much memory!(%d)\n",
			  MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof(wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		NDIS_UNLOCK(sc);
		ndis_m_start(sc);
		NDIS_LOCK(sc);
		len = 0;
		rval = ndis_set_info(sc, OID_802_11_BSSID_LIST_SCAN,
	    			NULL, &len);
		if (!rval) {	
			NDISDBG((CE_CONT,"scan command issued...\n"));
			outfp->wldp_result = WL_SUCCESS;
			NDIS_UNLOCK(sc);
			delay(300);
			NDIS_LOCK(sc);
		}else{
			NDISDBG((CE_WARN,"scan command issue failed\n"));
			outfp->wldp_result = WL_HW_ERROR;
			goto done;
		}	
		len = 8192;
		ess_list = (ndis_80211_bssid_list_ex *)
				kmem_zalloc(len, KM_NOSLEEP);
		if (ess_list == NULL) {
			NDISDBG((CE_CONT,
			     "ndis: can not alloc so much memory!(%d)\n",
			     len));
			return (ENOMEM);
		}
		rval = ndis_get_info(sc, OID_802_11_BSSID_LIST, ess_list, &len);
		NDISDBG((CE_CONT,"list len=%d, rval=%x\n",len,rval));
		if (rval) {
			NDISDBG((CE_CONT,
			    "get bss list failed: %d\n", rval));	
			kmem_free(ess_list, 8192);
			goto done;
		}

		wb = (ndis_wlan_bssid_ex *)&ess_list->nblx_bssid[0];

		for(i=0; i< ess_list->nblx_items; i++){
			wl_ess_conf_t *p_ess_conf;
			p_ess_conf = (wl_ess_conf_t *)
			    (buf + WIFI_BUF_OFFSET +
			    offsetof(wl_ess_list_t, wl_ess_list_ess) +
			     i * sizeof (wl_ess_conf_t));
			db = wb->nwbx_ssid.ns_ssid;
			NDIS11NDBG((CE_CONT,"SSID name %s\n",wb->nwbx_ssid.ns_ssid));
			bcopy(&wb->nwbx_ssid, &p_ess_conf->wl_ess_conf_essid,
			    sizeof(ndis_80211_ssid));
			bcopy(wb->nwbx_macaddr, p_ess_conf->wl_ess_conf_bssid, 6);
//			(p_ess_conf->wl_phy_conf).wl_phy_dsss_conf.wl_dsss_subtype = WL_DSSS;
			if(wb->nwbx_netinfra == NDIS_80211_NET_INFRA_IBSS)
				p_ess_conf->wl_ess_conf_bsstype = WL_BSS_IBSS;
			else if(wb->nwbx_netinfra == NDIS_80211_NET_INFRA_BSS)
				p_ess_conf->wl_ess_conf_bsstype = WL_BSS_BSS;
			//p_ess_conf->wl_ess_conf_authmode??
			p_ess_conf->wl_ess_conf_wepenabled = 
				(wb->nwbx_privacy)?WL_ENC_WEP : WL_NOENCRYPTION;
			NDIS11NDBG((CE_CONT, "wb->nwbx_privacy(none/wep) = %d\n", wb->nwbx_privacy));
			//wpa detection
			wpa_ie_parser(p_ess_conf, (uint8_t *)&wb->nwbx_ies[0], wb->nwbx_ielen);
			/*signal strength*/
			rssi = (wb->nwbx_rssi + 200)/12;
			if(rssi < 0)
				rssi = 0;
			else if(rssi > 15)
				rssi = 15;			
			p_ess_conf->wl_ess_conf_sl = rssi;
			//AP maxrate detect
			for (j = 0; j < sizeof(wb->nwbx_supportedrates); j++) {
		              if ((wb->nwbx_supportedrates[j] & 0x7f) >
		                  p_ess_conf->wl_supported_rates[0]) {
	                            p_ess_conf->wl_supported_rates[0] =
	                                         wb->nwbx_supportedrates[j] & 0x7f;
	                         }
	             	}
			NDIS11NDBG((CE_CONT, "maxrates = %d\n", p_ess_conf->wl_supported_rates[0]));
			//nettype
			NDIS11NDBG((CE_CONT,"wb->nwbx_nettype = 0x%x\n",wb->nwbx_nettype));
			ndis_nettype_parser(p_ess_conf, wb);
			wb = (ndis_wlan_bssid_ex *)((char *)wb + wb->nwbx_len);
		}
	
		((wl_ess_list_t *)(outfp->wldp_buf))->wl_ess_list_num = ess_list->nblx_items;
		outfp->wldp_length = WIFI_BUF_OFFSET +
			    offsetof(wl_ess_list_t, wl_ess_list_ess) +
	   		    (ess_list->nblx_items) * sizeof (wl_ess_conf_t);
		outfp->wldp_result = WL_SUCCESS;		
		
		kmem_free(ess_list, 8192);
	
	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_READONLY;
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;
}


static int32_t
ndis_cfg_bss_type(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char bssid[6];
	uint16_t i;
	uint32_t arg;
	int len;
	int rval;
	wldp_t *outfp;
	char *buf;
	uint16_t type=0;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			  "ieee80211: can not alloc so much memory!(%d)\n",
			  MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));
	outfp->wldp_length = WIFI_BUF_OFFSET + sizeof (wl_bss_type_t);
	
	if (cmd == WLAN_GET_PARAM) {
		len = sizeof(arg);
		rval = ndis_get_info(sc, OID_802_11_INFRASTRUCTURE_MODE, 
				&arg, &len);
		if(rval) {
			NDISDBG((CE_CONT,"get bss type failed: %d\n", rval));
		}
		
		if(arg == NDIS_80211_NET_INFRA_IBSS){
			type = WL_BSS_IBSS;
			isc->isc_opmode = IEEE80211_M_IBSS;
		}else if(arg == NDIS_80211_NET_INFRA_BSS){
			type = WL_BSS_BSS;
			isc->isc_opmode = IEEE80211_M_STA;
		}else if(arg == NDIS_80211_NET_INFRA_AUTO){
			type = WL_BSS_ANY;
			isc->isc_opmode = IEEE80211_M_STA;
		}else{
			NDISDBG((CE_CONT,"%s: Wrong infra mode 0x%x\n",
						__func__,arg));
		}

		(*(wl_bss_type_t *)(outfp->wldp_buf)) = type;
		outfp->wldp_result = WL_SUCCESS;
	} else if (cmd == WLAN_SET_PARAM) {
		type = (uint16_t)(*(wl_bss_type_t *)(infp->wldp_buf));
		if ((type != WL_BSS_BSS) &&
			(type != WL_BSS_IBSS) &&
			(type != WL_BSS_ANY)) {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}

		if(type == WL_BSS_IBSS) {
			isc->isc_opmode = IEEE80211_M_IBSS;
			arg = NDIS_80211_NET_INFRA_IBSS;
		}else if( type == WL_BSS_BSS) { 
			isc->isc_opmode = IEEE80211_M_STA;
			arg = NDIS_80211_NET_INFRA_BSS;
		}else if( type == WL_BSS_ANY){
			isc->isc_opmode = IEEE80211_M_STA;
			arg = NDIS_80211_NET_INFRA_BSS;
		}
		NDISDBG((CE_CONT,"%s: set bss type to 0x%x\n",
						__func__,type));		
		rval = ndis_set_info(sc, OID_802_11_INFRASTRUCTURE_MODE, 
				(void *)&arg, &len);
		if (rval)
			NDISDBG((CE_CONT, 
					"%s: set infra failed: %d\n", __func__, rval));
		outfp->wldp_result = WL_SUCCESS;	
	} else {
		cmn_err(CE_CONT,"%s:Missed set parm\n",__func__);
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_phy(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp = (wldp_t *)(mp->b_rptr);
	char bssid[6];
	uint16_t i;
	int rval=0;
	wldp_t *outfp;
	char *buf;
	uint32_t chan;
	int len;
	ndis_80211_config config;	
	ndis_wlan_bssid_ex	*bs=0;
	wl_dsss_t *dsss=0;
	wl_fhss_t *fhss=0;
	wl_ofdm_t *ofdm=0;
	wl_erp_t *erp=0;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			"ndis: can not alloc so much memory!(%d)\n",
			MAX_BUF_LEN));
		return (ENOMEM);
	}
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	outfp = (wldp_t *)buf;
	dsss = (wl_dsss_t *)(outfp->wldp_buf);
	fhss = (wl_fhss_t *)(outfp->wldp_buf);
	ofdm = (wl_ofdm_t *)(outfp->wldp_buf);
	erp = (wl_erp_t *)(outfp->wldp_buf);
	if (cmd == WLAN_GET_PARAM) {
		if ((rval = ndis_get_assoc(sc, &bs)) == 0) {
			NDISCFGPHY((CE_CONT, "%s: get assoc ok, type %d\n",
				__func__,bs->nwbx_nettype));
			switch(bs->nwbx_nettype) {
			case NDIS_80211_NETTYPE_11FH:
				outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_fhss_t);
				fhss->wl_fhss_subtype = WL_FHSS;
				fhss->wl_fhss_dwelltime = bs->nwbx_config.nc_fhconfig.ncf_dwelltime;
				fhss->wl_fhss_channel = ieee80211_mhz2ieee(bs->nwbx_config.nc_dsconfig / 1000, 0);
				break;
			case NDIS_80211_NETTYPE_11DS:
				outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_dsss_t);
				dsss->wl_dsss_subtype = WL_DSSS;
				dsss->wl_dsss_channel = ieee80211_mhz2ieee(bs->nwbx_config.nc_dsconfig  / 1000, 0);
				break;
			case NDIS_80211_NETTYPE_11OFDM5:
				outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_ofdm_t);
				ofdm->wl_ofdm_subtype = WL_OFDM;		
				ofdm->wl_ofdm_frequency = bs->nwbx_config.nc_dsconfig  / 1000;
				break;
			case NDIS_80211_NETTYPE_11OFDM24:
				outfp->wldp_length = WIFI_BUF_OFFSET + sizeof(wl_erp_t);
				erp->wl_erp_subtype = WL_ERP;		
				erp->wl_erp_channel = ieee80211_mhz2ieee(bs->nwbx_config.nc_dsconfig  / 1000, 0);
				break;
			default:
				NDISCFGPHY((CE_CONT, "unknown nettype %d\n",
					bs->nwbx_nettype));
				break;
			}
			kmem_free(bs,bs->nwbx_len);
			outfp->wldp_result = WL_SUCCESS;
		}else{
			NDISCFGPHY((CE_CONT, "%s: get assoc failed\n",
				__func__));
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
	} else if (cmd == WLAN_SET_PARAM) {
		if(sc->ndis_link){
			NDISDBG((CE_CONT,"%s: don't set phy when link up\n",__func__));
			kmem_free(buf, MAX_BUF_LEN);
			return (EINVAL);
		}
			
		chan = (uint32_t)
			(((wl_phy_conf_t *)(infp->wldp_buf))->wl_phy_dsss_conf.wl_dsss_channel);		
		NDISDBG((CE_CONT,"%s: chan set to [%d]\n",__func__, chan)); 
		if (chan < 1 || chan > IEEE80211_CHAN_MAX) {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}

		len = sizeof(config);
		bzero((char *)&config, len);
		config.nc_length = len;
		config.nc_fhconfig.ncf_length = sizeof(ndis_80211_config_fh);
		rval = ndis_get_info(sc, OID_802_11_CONFIGURATION, &config, &len); 

		/*
		 * Some drivers expect us to initialize these values, so
		 * provide some defaults.
		 */
		if (config.nc_beaconperiod == 0)
			config.nc_beaconperiod = 100;
		if (config.nc_atimwin == 0)
			config.nc_atimwin = 100;
		if (config.nc_fhconfig.ncf_dwelltime == 0)
			config.nc_fhconfig.ncf_dwelltime = 200;

		if (rval == 0 ) { //workaround
			int chanflag = 0;

			if (isc->isc_sup_rates[IEEE80211_MODE_11G].rs_nrates)
				chanflag |= IEEE80211_CHAN_G;
			if (chan <= 14)
				chanflag |= IEEE80211_CHAN_B;
			if (isc->isc_sup_rates[IEEE80211_MODE_11A].rs_nrates &&
			    chan > 14)
				chanflag = IEEE80211_CHAN_A;
		
			NDISDBG((CE_CONT,"%s: try channel fixed to [%d]\n",__func__,chan));	
			isc->isc_ibss_chan = &isc->isc_channels[chan]; //fix channel pointer
			NDISDBG((CE_CONT,"%s: channel fixed to [%d]\n",__func__,chan));	
			isc->isc_ibss_chan->ich_freq =
			    ieee80211_ieee2mhz(chan, chanflag);
			config.nc_dsconfig =
			    isc->isc_ibss_chan->ich_freq * 1000;
			isc->isc_bss->ni_chan = isc->isc_ibss_chan;
			len = sizeof(config);
			config.nc_length = len;
			config.nc_fhconfig.ncf_length =
				    sizeof(ndis_80211_config_fh);
			rval = ndis_set_info(sc, OID_802_11_CONFIGURATION,
				    (void *)&config, &len);
			if (rval){
				NWAMDBG((CE_CONT,"%s: failed\n",__func__));
				NDISDBG((CE_CONT, "couldn't change "
					    "DS config to %ukHz: %d\n",
					    config.nc_dsconfig, rval));
				outfp->wldp_result = WL_NOTSUPPORTED;
			}else{
				NDISDBG((CE_CONT, "changed "
					    "DS config to %ukHz: %d\n",
					    config.nc_dsconfig, rval));
			}
		} else 
			NDISDBG((CE_CONT, "couldn't retrieve "
			    "channel info: %d\n", rval));
		
		outfp->wldp_result = WL_SUCCESS;	
	} else {
		NWAMDBG((CE_CONT,"%s: should be command\n",__func__));
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}
done:
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}


static int32_t
ndis_cfg_rates(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint8_t rates[16];
	int  i,k;
	uint32_t arg;
	int rval;
	wldp_t *outfp;
	char *buf;
	int len=16;
	uint8_t desire_rate=0;
	int rates_num=0, desire_rate_nr=0;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		bzero(rates, 16);
		rval = ndis_get_info(sc, OID_802_11_DESIRED_RATES, 
				(void *)rates, &len);
		if(rval) 
			NDISRATES((CE_CONT,"get in working rates failed: %d\n", rval));
		
		for(i=0; i < 16; i++) {
			NDISRATES((CE_CONT, "list rates[%x]=%d;\n",i, (rates[i] & 0x7f)));
			/*
			 * a desired data rate in units of 0.5 Mbps 
			 */
			switch (rates[i] & 0x7f ) {
				case 0:
					break;
				case 2:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_1M;
					break;
				case 4:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_2M;
					break;
				case 11:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_5_5M;
					break;
				case 12:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_6M;
					break;
				case 18:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_9M;
					break;
				case 22:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_11M;
					break;
				case 24:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_12M;
					break;
				case 36:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_18M;
					break;
				case 44:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_22M;
					break;
				case 48:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_24M;
					break;
				case 66:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_33M;
					break;
				case 72:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_36M;
					break;
				case 96:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_48M;
					break;
				case 108:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_54M;
					break;
				default:
					NDISRATES((CE_CONT, "unknow rates[%x]=%d;\n",i, (rates[i] & 0x7f)));
					break;
			}
		}

		((wl_rates_t *)(outfp->wldp_buf))->wl_rates_num = rates_num;
		outfp->wldp_length = WIFI_BUF_OFFSET +
				offsetof(wl_rates_t, wl_rates_rates) +
				rates_num * sizeof (char);
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
		desire_rate_nr =  ((wl_rates_t *)(infp->wldp_buf))->wl_rates_num;
		if(desire_rate_nr > 1){
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
		desire_rate = (((wl_rates_t *)(infp->wldp_buf))->wl_rates_rates)[0];
		NDISRATES((CE_CONT, "ndis: set tx_rate = %d\n", desire_rate));
		bzero(rates, 16);
		rval = ndis_get_info(sc, OID_802_11_SUPPORTED_RATES, 
				(void *)rates, &len);
		for (i = 0; i < len; i++) {
			if (rates[i] & 0x80)
				continue;
			if ((rates[i] & 0x7f) > desire_rate) {
				NDISRATES((CE_CONT, "ndis: clear rate[%d]: %d to 0\n", i, rates[i]));
				rates[i] = 0;
			}
		}
		rval = ndis_set_info(sc, OID_802_11_DESIRED_RATES, 
				(void *)rates, &len);
		if(rval) {
			NDISRATES((CE_CONT, "couldn't set "
		    		"rates info: %d\n", rval));
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_SUCCESS;
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_supported_rates(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint8_t rates[16];
	uint16_t i;
	uint32_t arg;
	int rval;
	int len=16;
	int rates_num=0;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ieee80211: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		bzero(rates, len);
		rval = ndis_get_info(sc, OID_802_11_SUPPORTED_RATES, 
				(void *)rates, &len);
		if(rval) {
			NDISDBG((CE_CONT,
			    "get support rates failed: %d\n", rval));	
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
	
		for(i=0; i< 16; i++) {
			NDISRATES((CE_CONT, "list rates[%x]=%d;\n",i, (rates[i] & 0x7f)));
			switch (rates[i] & 0x7F){
				case 0:
					break;
				case 2:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_1M;
					break;
				case 4:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_2M;
					break;
				case 11:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_5_5M;
					break;
				case 12:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_6M;
					break;
				case 36:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_18M;
					break;
				case 44:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_22M;
					break;
				case 48:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_24M;
					break;
				case 66:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_33M;
					break;
				case 72:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_36M;
					break;
				case 96:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_48M;
					break;
				case 108:
					(((wl_rates_t *)
					  (outfp->wldp_buf))->wl_rates_rates)[rates_num++] 
						= WL_RATE_54M;
					break;
				default:
					NDISDBG((CE_CONT, "unknow rates %d\n", (rates[i] & 0x7f)));
					break;
			}
		}
		((wl_rates_t *)(outfp->wldp_buf))->wl_rates_num = rates_num;
		outfp->wldp_length = WIFI_BUF_OFFSET +
			offsetof(wl_rates_t, wl_rates_rates) +
			rates_num * sizeof (char);
		
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_NOTSUPPORTED;	
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_power_mode(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	char bssid[6];
	uint16_t i,powermode=0;
	uint32_t arg;
	int rval;
	int len;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		len = sizeof(arg);
		rval = ndis_get_info(sc, OID_802_11_POWER_MODE, &arg, &len);

		if (rval)
			NDISDBG((CE_CONT,
			    "get power mode failed: %d\n", rval));

		switch(arg){
		case NDIS_80211_POWERMODE_CAM:
			NDISDBG((CE_CONT,"%s:NDIS_80211_POWERMODE_CAM\n ",__func__));
			powermode = WL_PM_AM;
			break;
		case NDIS_80211_POWERMODE_MAX_PSP:
			NDISDBG((CE_CONT,"%s:NDIS_80211_POWERMODE_MAX_PSP\n ",__func__));
			powermode = WL_PM_MPS;
			break;
		case NDIS_80211_POWERMODE_FAST_PSP:
			NDISDBG((CE_CONT,"%s:NDIS_80211_POWERMODE_FAST_PSP\n ",__func__));
			powermode = WL_PM_FAST;
			break;
		}
					
		if (powermode == WL_PM_AM)
			isc->isc_flags &= ~IEEE80211_F_PMGTON;
		else
			isc->isc_flags |= IEEE80211_F_PMGTON;
		
		((wl_ps_mode_t *)(outfp->wldp_buf))->wl_ps_mode = powermode;

		outfp->wldp_length = WIFI_BUF_OFFSET + 
			              sizeof (wl_ps_mode_t);
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
/*		if(sc->ndis_link){	
			cmn_err(CE_WARN,"%s: don't set powermode when link up\n",__func__);
			kmem_free(buf, MAX_BUF_LEN);
			return (EINVAL);
		}
*/			
		powermode = (uint16_t)(((wl_ps_mode_t *)(infp->wldp_buf))->wl_ps_mode);
		if ((powermode != WL_PM_AM) &&
			(powermode != WL_PM_MPS) &&
			(powermode != WL_PM_FAST)) {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
		switch(powermode){
		case WL_PM_AM:
			arg = NDIS_80211_POWERMODE_CAM;
			break;
		case WL_PM_MPS:
			arg = NDIS_80211_POWERMODE_MAX_PSP;
			break;
		case WL_PM_FAST:
			arg = NDIS_80211_POWERMODE_FAST_PSP;
			break;
		}
		
		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_POWER_MODE, 
				(void *)&arg, &len);
		if (rval){
			NDISDBG((CE_CONT,"set power failed: %d\n",rval)); 
			kmem_free(buf, MAX_BUF_LEN);
			return (EINVAL);
		}else{
			if (arg == NDIS_80211_POWERMODE_CAM)
				isc->isc_flags &= ~IEEE80211_F_PMGTON;
			else
				isc->isc_flags |= IEEE80211_F_PMGTON;
		}
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_SUCCESS;	
	} else {
		NDISDBG((CE_CONT,"%s: powermode failed\n",__func__));			
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}


static int32_t
ndis_cfg_auth_mode(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint16_t i;
	uint32_t	arg;
	int rval;
	int len;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		len = sizeof(arg);
		rval = ndis_get_info(sc, OID_802_11_AUTHENTICATION_MODE,
				&arg, &len);
		if (rval)
			NDISINF((CE_CONT,"get authmode failed: %d\n", rval));

		isc->isc_authmode = arg;

		if( arg == NDIS_80211_AUTHMODE_OPEN) {
			*(wl_authmode_t *)(outfp->wldp_buf) = WL_OPENSYSTEM;
			NDISDBG((CE_CONT,"WL_OEPNSYS mode\n"));
		}else if( arg == NDIS_80211_AUTHMODE_SHARED) {
			*(wl_authmode_t *)(outfp->wldp_buf) = WL_SHAREDKEY;
			NDISDBG((CE_CONT,"WL_SHAREDKEY mode\n"));
		}else if( (arg == NDIS_80211_AUTHMODE_WPA) ||
			  (arg == NDIS_80211_AUTHMODE_WPANONE) ||
			  (arg == NDIS_80211_AUTHMODE_WPAPSK) ||
			  (arg == NDIS_80211_AUTHMODE_AUTO) ){
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_HW_ERROR;
			NDISDBG((CE_NOTE,"Known authmodes in H/W :0x%8x\n", arg));
			goto done;
		}else {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_HW_ERROR;
			NDISDBG((CE_NOTE,"Unknown authmodes in H/W :0x%8x\n", arg));
			goto done;
		}

		outfp->wldp_length = WIFI_BUF_OFFSET + 
			              sizeof (wl_authmode_t);
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
		arg = *(uintptr_t *)((wl_authmode_t *)(infp->wldp_buf));
		if ((arg != WL_OPENSYSTEM) &&
			(arg != WL_SHAREDKEY)) {
			NDISINF((CE_NOTE,"What's the parameter? : %d\n",arg));
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
		
		switch(arg){
		case WL_OPENSYSTEM:
			arg = NDIS_80211_AUTHMODE_OPEN;			
			NDISDBG((CE_NOTE,"set auth to NDIS_80211_AUTHMODE_OPEN\n"));
			break;
		case WL_SHAREDKEY:
			arg = NDIS_80211_AUTHMODE_SHARED;			
			NDISDBG((CE_NOTE,"set auth to NDIS_80211_AUTHMODE_SHARED\n"));
			break;
		}

		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_AUTHENTICATION_MODE, 
				(void *)&arg, &len);
		if (rval){
			NDISINF((CE_CONT, "set auth mode failed: %d\n", rval));
			kmem_free(buf, MAX_BUF_LEN);
			return (EINVAL);
		}else{
			isc->isc_authmode = arg;
		}
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_SUCCESS;	
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}


static int32_t
ndis_cfg_create_ibss(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint16_t i;
	uint32_t arg;
	int rval;
	int ret;
	wldp_t *outfp;
	char *buf;
	int len;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		if(isc->isc_opmode == IEEE80211_M_IBSS)
			*(wl_create_ibss_t *)(outfp->wldp_buf) = B_TRUE;
		else if(isc->isc_opmode == IEEE80211_M_STA)
			*(wl_create_ibss_t *)(outfp->wldp_buf) = B_FALSE;
		else
			NDISDBG((CE_CONT,"%s: what's bss type now?\n",__func__));

		outfp->wldp_length = WIFI_BUF_OFFSET +
					sizeof(wl_create_ibss_t); 
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
		ret = (uint16_t)(*(wl_create_ibss_t *)(infp->wldp_buf));
		if (ret != 0 && ret != 1) {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
			goto done;
		}
	
		if(ret){
			arg = NDIS_80211_NET_INFRA_IBSS;
			isc->isc_opmode = IEEE80211_M_IBSS;
		}else{
			arg = NDIS_80211_NET_INFRA_BSS;
			isc->isc_opmode = IEEE80211_M_STA;
		}
		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_INFRASTRUCTURE_MODE, 
				(void *)&arg, &len);
		if (rval){
			NDISDBG((CE_CONT,"create ibss failed: %d\n", rval));
		}else
			NDISDBG((CE_CONT,"%s: create ibss successfully\n",__func__));
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_SUCCESS;	
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_rssi(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	ndis_wlan_bssid_ex	*bs = NULL;
	int i;
	int rval;
	wldp_t *outfp;
	char *buf;
	uint32_t 	rssi;
	
	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		if(sc->ndis_link){
			rval = ndis_get_assoc(sc, &bs);
			if (rval){
				NDISDBG((CE_CONT, 
					"get rssi failed: %d\n", rval));
				kmem_free(buf, MAX_BUF_LEN);
				return (EINVAL);
			}else{
				rssi = (bs->nwbx_rssi + 200)/12;
				if(rssi < 0)
					rssi = 0;
				else if(rssi > 15)
					rssi = 15;
				*(wl_rssi_t *)(outfp->wldp_buf) = rssi;
				kmem_free(bs,bs->nwbx_len);
			}
		}else
			*(wl_rssi_t *)(outfp->wldp_buf) = 0;
		outfp->wldp_length = WIFI_BUF_OFFSET +
					sizeof(wl_rssi_t); 
		outfp->wldp_result = WL_SUCCESS;

	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_READONLY;	
	} else {
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int
ndis_cfg_linkstatus(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	uint16_t i;
	wldp_t *outfp;
	char *buf;
	int iret;
	ieee80211com_t *isc = (ieee80211com_t *)&(sc->ic);

	buf = kmem_zalloc(MAX_BUF_LEN, KM_SLEEP);
	outfp = (wldp_t *)buf;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	outfp->wldp_length = WIFI_BUF_OFFSET + sizeof (wl_linkstatus_t);
	outfp->wldp_result = WL_SUCCESS;

	if (cmd == WLAN_GET_PARAM) {
		if (!NDIS_INITIALIZED(sc)){
			*(wl_linkstatus_t *)(outfp->wldp_buf) = WL_NOTCONNECTED;
		}else{
			delay(100);	
			*(wl_linkstatus_t *)(outfp->wldp_buf) =
		    		(IEEE80211_S_RUN == isc->isc_state) ?
		    			WL_CONNECTED : WL_NOTCONNECTED;
		}
	} else if (cmd == WLAN_SET_PARAM) {
			outfp->wldp_length = WIFI_BUF_OFFSET;
			outfp->wldp_result = WL_NOTSUPPORTED;
	}

	for (i = 0; i < (outfp->wldp_length); i++)
		mi_mpprintf_putc((char *)mp, buf[i]);
	iret = (int)(outfp->wldp_result);
	kmem_free(buf, MAX_BUF_LEN);
	return (iret);
}

static int32_t
ndis_cfg_radio(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	uint16_t i;
	int iret;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_NOTE, "can not alloc so much memory!(%d)\n",
		    MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		*(wl_radio_t *)(outfp->wldp_buf) = B_TRUE;
		outfp->wldp_length = WIFI_BUF_OFFSET + sizeof (wl_radio_t);
		outfp->wldp_result = WL_SUCCESS;
	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_NOTSUPPORTED;
	} else {
		NWAMDBG((CE_CONT,"%s: command error\n",__func__));
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}
	for (i = 0; i < (outfp->wldp_length); i++)
		(void) mi_mpprintf_putc((char *)mp, buf[i]);
	kmem_free(buf, MAX_BUF_LEN);
	return (0);
}

static int32_t
ndis_cfg_disconnect(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	wldp_t *infp;
	int i;
	int rval;
	uint32_t value=0;
	int len=4;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	if(mp) {
	    infp = (wldp_t *)mp->b_rptr;
	    bcopy(mp->b_rptr, buf,  sizeof (wldp_t));
	}
	outfp->wldp_length = WIFI_BUF_OFFSET; 
		
	if (cmd == WLAN_COMMAND) {
		NDISDBG((CE_CONT,"SET in disconnect\n"));
		rval = ndis_set_info(sc, OID_802_11_DISASSOCIATE, (void *)&value, &len);

		if (rval){
			NDISDBG((CE_CONT, "disassociate failed: %d\n", rval));
			goto done;
		}else{
			NDISDBG((CE_CONT,"Disconnect successfully!\n"));
			NDIS_UNLOCK(sc);
			ndis_m_stop(sc);
			ndis_m_start(sc);
			NDIS_LOCK(sc);
		}

		sc->ic.isc_state = IEEE80211_S_INIT;
		sc->ic.isc_flags &= ~IEEE80211_F_WEPON;
		sc->ic.isc_authmode = NDIS_80211_AUTHMODE_OPEN;
		sc->ic.isc_opmode = IEEE80211_M_STA;
		memset(&sc->ic.isc_des_essid, 0, IEEE80211_NWID_LEN);
		sc->ic.isc_des_esslen = 0;
		sc->ic.isc_wep_txkey = 0;
		memset(&sc->ic.isc_bss->ni_essid,0,IEEE80211_NWID_LEN);
		sc->ic.isc_bss->ni_esslen = 0;
		for (i = 0; i < IEEE80211_WEP_NKID; i++) {
			sc->ic.isc_nw_keys[i].wk_len = 0;
			bzero(&sc->ic.isc_nw_keys[i].wk_key,IEEE80211_KEYBUF_SIZE);
		}

		outfp->wldp_result = WL_SUCCESS;
	}else
		NWAMDBG((CE_CONT,"%s, not WLAN_COMMAND,error\n",__func__));
done:
	if(mp) {
	    for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	    }
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_loaddefault(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint16_t i;
	uint32_t arg;
	int rval;
	uint32_t len;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET; 
		outfp->wldp_result = WL_NOTSUPPORTED;
		NWAMDBG((CE_CONT,"%s: What do you want to get?\n",__func__));
	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_NOTSUPPORTED;
		NWAMDBG((CE_CONT,"%s, What do you want to set?\n",__func__));
	} else if(cmd == WLAN_COMMAND){ 
		len = sizeof(arg);
		arg = NDIS_80211_RELOADDEFAULT_WEP;
		rval = ndis_set_info(sc, OID_802_11_RELOAD_DEFAULTS, 
				(void *)&arg, &len);
		if (rval)
			NDISINF((CE_CONT,"reload default failed: %d\n", rval)); 
		NWAMDBG((CE_CONT,"%s: WLAN_COMMAND\n",__func__));
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_SUCCESS;		
	}else{
		NWAMDBG((CE_CONT,"%s: subcommand = 0x%x\n",
			__func__,cmd));
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
						    
}

static int32_t
ndis_cfg_nodename(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{
	ieee80211com_t *isc = (ieee80211com_t *)&sc->ic;
	wldp_t *infp;
	uint16_t i;
	uint32_t arg;
	int rval;
	uint32_t len;
	wldp_t *outfp;
	char *buf;

	buf = kmem_zalloc(MAX_BUF_LEN, KM_NOSLEEP);
	if (buf == NULL) {
		NDISDBG((CE_CONT,
			   "ndis: can not alloc so much memory!(%d)\n",
			   MAX_BUF_LEN));
		return (ENOMEM);
	}
	outfp = (wldp_t *)buf;

	infp = (wldp_t *)mp->b_rptr;
	bcopy(mp->b_rptr, buf,  sizeof (wldp_t));

	if (cmd == WLAN_GET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET; 
		outfp->wldp_result = WL_NOTSUPPORTED;
		NWAMDBG((CE_CONT,"%s: What do you want to get?\n",__func__));
	} else if (cmd == WLAN_SET_PARAM) {
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_NOTSUPPORTED;
		NWAMDBG((CE_CONT,"%s, What do you want to set?\n",__func__));
	} else if(cmd == WLAN_COMMAND) { 
		outfp->wldp_length = WIFI_BUF_OFFSET;
		outfp->wldp_result = WL_NOTSUPPORTED;
		NWAMDBG((CE_CONT,"%s, What command do you want to send?\n",__func__));
	} else {
		NWAMDBG((CE_CONT,"%s: subcommand = 0x%x\n",
			__func__,cmd));
		kmem_free(buf, MAX_BUF_LEN);
		return (EINVAL);
	}

done:	
	for (i = 0; i < (outfp->wldp_length); i++) {
		mi_mpprintf_putc((char *)mp, buf[i]);
	}
	kmem_free(buf, MAX_BUF_LEN);	
	return WL_SUCCESS;	
}

static int32_t
ndis_getset(struct ndis_softc *sc, mblk_t *mp, uint32_t cmd)
{

	int ret = WL_SUCCESS;
	NDISIOCTL((CE_CONT,"--> %s :cmd = %s, subcmd == 0x%x\n",
				__func__, (cmd == WLAN_GET_PARAM)?"get":"set", ((wldp_t *)mp->b_rptr)->wldp_id));	

	if(sc->ndis_suspended)
		return EINVAL;
	switch(((wldp_t *)mp->b_rptr)->wldp_id) {
		case WL_SCAN:
			NDISIOCTL((CE_CONT,"ndis: WL_SCAN_DO\n"));
			ret = ndis_cfg_scan(sc, mp, cmd);
			break;
		case WL_ESSID:
			NDISIOCTL((CE_CONT,"ndis: WL_ESSID\n"));
			ret = ndis_cfg_essid(sc, mp, cmd);
			break;
		case WL_BSSID:
			NDISIOCTL((CE_CONT,"ndis: WL_BSSID\n"));
			ret = ndis_cfg_bssid(sc, mp, cmd);
			break;
		case WL_ENCRYPTION:
			NDISIOCTL((CE_CONT,"ndis: WL_ENCRYPTION\n"));
			ret = ndis_cfg_encryption(sc, mp, cmd);
			break;
		case WL_WEP_KEY_TAB:
			NDISIOCTL((CE_CONT,"ndis: WL_WEP_KEY_TAB\n"));
			ret = ndis_cfg_setwepkey(sc, mp, cmd);
			break;
		case WL_WEP_KEY_ID:
			NDISIOCTL((CE_CONT,"ndis: WL_WEP_KEY_ID\n"));
			ret = ndis_cfg_wepkeyindex(sc, mp, cmd);
			break;
		case WL_ESS_LIST:
			NDISIOCTL((CE_CONT,"ndis: WL_ESS_LIST\n"));
			ret = ndis_cfg_esslist(sc, mp, cmd);
			break;
		case WL_AUTH_MODE:
			NDISIOCTL((CE_CONT,"ndis: WL_AUTH_MODE\n"));
			ret = ndis_cfg_auth_mode(sc, mp, cmd); 
			break;
		case WL_POWER_MODE:
			NDISIOCTL((CE_CONT,"ndis: WL_POWER_MODE\n"));
			ret = ndis_cfg_power_mode(sc, mp, cmd); 
			break;
		case WL_BSS_TYPE:
			NDISIOCTL((CE_CONT,"ndis: WL_BSS_TYPE\n"));
			ret = ndis_cfg_bss_type(sc, mp, cmd); 
			break;
		case WL_PHY_CONFIG:
			NDISIOCTL((CE_CONT,"ndis: WL_PHY_CONFIG\n"));
			ret = ndis_cfg_phy(sc, mp, cmd); 
			break;
		case WL_DESIRED_RATES:
			NDISIOCTL((CE_CONT,"ndis: WL_DESIRED_RATES\n"));
			ret = ndis_cfg_rates(sc, mp, cmd);
			break;
		case WL_SUPPORTED_RATES:
			NDISIOCTL((CE_CONT,"ndis: WL_SUPPORTED_RATES\n"));
			ret = ndis_cfg_supported_rates(sc, mp, cmd); 
			break;
		case WL_CREATE_IBSS:
			NDISIOCTL((CE_CONT,"ndis: WL_CREATE_IBSS\n"));
			ret = ndis_cfg_create_ibss(sc, mp, cmd); 
			break;
		case WL_RSSI:
			NDISIOCTL((CE_CONT,"ndis: WL_RSSI\n"));
			ret = ndis_cfg_rssi(sc, mp, cmd); 
			break;
		case WL_LINKSTATUS:
			NDISIOCTL((CE_CONT,"ndis: WL_LINKSTATUS\n"));
			ret = ndis_cfg_linkstatus(sc, mp, cmd);
			break;
		case WL_RADIO:
			NDISIOCTL((CE_CONT,"ndis: WL_RADIO\n"));
			ret = ndis_cfg_radio(sc, mp, cmd); 
			break;
		case WL_DISASSOCIATE:
			NDISIOCTL((CE_CONT,"ndis: WL_DISASSOCIATE\n"));
			ret = ndis_cfg_disconnect(sc,mp,cmd);
			break;
		case WL_LOAD_DEFAULTS:
			NDISIOCTL((CE_CONT,"ndis: WL_LOAD_DEFAULTS\n"));
			ret = ndis_cfg_loaddefault(sc, mp, cmd); 
			break;
		case WL_NODE_NAME:
			NDISIOCTL((CE_CONT,"ndis: WL_NODE_NAME\n"));
			ret = ndis_cfg_nodename(sc, mp, cmd);
			break;
		default:
			NDISIOCTL((CE_CONT,"MISTAKE 0x%x",((wldp_t *)mp->b_rptr)->wldp_id));
			break;
	}
	NDISIOCTL((CE_CONT,"<--ndis_getset: cmd = 0x%x, subcmd = 0x%x, ret = 0x%x\n",
				cmd, ((wldp_t *)mp->b_rptr)->wldp_id, ret));
	return ret;
}


static void
ndis_wlan_ioctl(struct ndis_softc *sc, queue_t *wq, mblk_t *mp, uint32_t cmd)
{
	struct	iocblk	*iocp = (struct iocblk *)mp->b_rptr;
	wldp_t 	*infp;
	uint32_t len, ret;
	mblk_t	*mp1;

	/*
	 * sanity check
	 */
	if (iocp->ioc_count == 0 || !(mp1 = mp->b_cont)) {
		miocnak(wq, mp, 0, EINVAL);
		return;
	}

	/*
	 * assuming single data block
	 */
	if (mp1->b_cont) {
		freemsg(mp1->b_cont);
		mp1->b_cont = NULL;
	}

	/*
	 * we will overwrite everything
	 */
	mp1->b_wptr = mp1->b_rptr;

	infp = (wldp_t *)mp1->b_rptr;

	NDIS_LOCK(sc);
	ret = ndis_getset(sc, mp1, cmd);
	NDIS_UNLOCK(sc);
	len = msgdsize(mp1);
	miocack(wq, mp, len, ret);

}

static void ndis_m_ioctl(void *arg, queue_t *wq, mblk_t *mp)
{
	struct iocblk *iocp;
	uint32_t cmd, ret;
	struct ndis_softc *sc = (struct ndis_softc *)arg;
	boolean_t need_privilege = B_TRUE;

	/*
	 * Validate the command before bothering with the mutexen ...
	 */
	iocp = (struct iocblk *)mp->b_rptr;
	iocp->ioc_error = 0;
	cmd = iocp->ioc_cmd;
	switch (cmd) {
	default:
		NDISDBG((CE_CONT, "ndis_m_ioctl: unknown cmd 0x%x", cmd));
		miocnak(wq, mp, 0, EINVAL);
		return;
	case WLAN_GET_PARAM:
		need_privilege = B_TRUE;
		break;
	case WLAN_SET_PARAM:
	case WLAN_COMMAND:
		break;
	}
	/*
	 * Check net_config privilege
	 */
	if (need_privilege) {
		if (ret = secpolicy_net_config(iocp->ioc_cr, B_FALSE)) {
			miocnak(wq, mp, 0, ret);
			return;
		}
	}
	ndis_wlan_ioctl(sc, wq, mp, cmd);
}



/*
 * Main transmit routine. To make NDIS drivers happy, we need to
 * transform mblk chains into NDIS packets and feed them to the
 * send packet routines. Most drivers allow you to send several
 * packets at once (up to the maxpkts limit). Unfortunately, rather
 * that accepting them in the form of a linked list, they expect
 * a contiguous array of pointers to packets.
 *
 * For those drivers which use the NDIS scatter/gather DMA mechanism,
 * we need to perform busdma work here. Those that use map registers
 * will do the mapping themselves on a buffer by buffer basis.
 */

static int
ndis_tx_begin(struct ndis_softc	*sc, mblk_t *mp)
{
	ndis_packet *p = NULL;
	int status, pcnt=0;
	mblk_t *mpx;

	if (mp == NULL){
		cmn_err(CE_CONT,"%s:You are NULL!\n",__func__);
		return (GLD_FAILURE);
	}

	if( pullupmsg(mp, -1) == 0){
		freemsg(mp);
		return GLD_FAILURE;
	}
	if(sc->ndis_txpending ){ 
		mpx = mac_wifi_header_backout(mp, NULL);
	
		if (ndis_mtop(mpx, &sc->ndis_txarray[sc->ndis_txidx])) {
			NDISDBG((CE_CONT,"P5, idx[%d]alloced failed\n",sc->ndis_txidx));
			return(GLD_FAILURE);
		}

		/*
	 	* Save pointer to original mblk
	 	* so we can free it later.
	 	*/
		p = sc->ndis_txarray[sc->ndis_txidx];
		p->np_txidx = sc->ndis_txidx;
		p->np_m0 = mpx;
		p->np_oob.npo_status = NDIS_STATUS_PENDING;
		
		NDIS_INC(sc);
		sc->ndis_txpending--; 
		pcnt++;
	}
	if(pcnt == 0){
		NDISDBG((CE_CONT,"No slot for tx\n"));
		return (GLD_FAILURE);
	}
	if (sc->ndis_maxpkts == 1){
		ndis_send_packet(sc, p);
	}else
		ndis_send_packets(sc, &p, 0x1);
	return(GLD_SUCCESS);
}

static int32_t
ndis_m_setmulti(void *arg, boolean_t on, const uint8_t *mca)
{

	struct ndis_softc *sc = (struct ndis_softc *)arg;
	uint32_t	retval, len;
	uint32_t	mclist_len, mclistsz;
	uint8_t	*mclist;
	uint8_t	*mc_p;
	uint32_t	packet_filter;
	int	i;

	NDIS_LOCK(sc);
	if(sc->ndis_suspended || !NDIS_INITIALIZED(sc)){
		NDIS_UNLOCK(sc);
		return EINVAL;
	}
	if(!sc->sc_mcast) {	
		/*
		 * first init
		 */
		len = sizeof(mclistsz);
		ndis_get_info(sc, OID_802_3_MAXIMUM_LIST_SIZE, &mclistsz, &len);
	//	NDISINF((CE_CONT,"mclistsz=%d\n",mclistsz));
		mclist_len = ETHER_ADDR_LEN * mclistsz;
		sc->sc_mcast = kmem_alloc(mclist_len, KM_NOSLEEP);

		if (sc->sc_mcast == NULL) {
			NDISINF((CE_CONT,"NULL mcast\n"));
			sc->ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
			goto out;
		}
		bzero(sc->sc_mcast, mclist_len);
		sc->sc_mcast_count = 0;
		sc->sc_mcast_len = mclist_len;
	}

	mc_p = sc->sc_mcast;
	if(on) {
		for(i=0; i<sc->sc_mcast_count; i++, mc_p+= (ETHER_ADDR_LEN)) {
			if(!ether_cmp(mca, mc_p))
				break;
		}
		if(i < (sc->sc_mcast_count))  /* find the entry */
			goto out;
		
		mc_p = sc->sc_mcast;

		/* find a free entry */
		if( (sc->sc_mcast_count + 1) > (sc->sc_mcast_len / ETHER_ADDR_LEN))
			goto out;
		mc_p += (ETHER_ADDR_LEN ) * (sc->sc_mcast_count);
		ether_copy(mca, mc_p);
		sc->sc_mcast_count ++;
	}else {
		for (i = 0; i < sc->sc_mcast_count; i++, mc_p += (ETHER_ADDR_LEN )) {
			if (!ether_cmp(mca, mc_p))
				break;
		}
		if (i >= sc->sc_mcast_count)  /* not found */
			goto out;

		/*
		 * delete the entry
		 */
		bzero(mc_p,ETHER_ADDR_LEN);
		sc->sc_mcast_count --;
	}

	sc->ndis_filter |= NDIS_PACKET_TYPE_MULTICAST;

	len = (sc->sc_mcast_count) * ETHER_ADDR_LEN;
	retval = ndis_set_info(sc, OID_802_3_MULTICAST_LIST, sc->sc_mcast, &len);
	if (retval) {
		NDISINF((CE_CONT, "set mclist failed: %d\n", retval));
		sc->ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		sc->ndis_filter &= ~NDIS_PACKET_TYPE_MULTICAST;
	}
	
out:
	packet_filter = sc->ndis_filter;
	len = sizeof(packet_filter);
	retval = ndis_set_info(sc, OID_GEN_CURRENT_PACKET_FILTER,
	    		&packet_filter, &len);
	if (retval) {
		NDISINF((CE_CONT, "set filter failed: %d\n", retval));
		NDIS_UNLOCK(sc);
		return (NDIS_FAILURE);
	}
	NDIS_UNLOCK(sc);
	return (NDIS_SUCCESS);

}

static int
ndis_m_prom(void *arg, boolean_t on)
{
	uint32_t		i, error, len,err;
	struct ndis_softc *sc = (struct ndis_softc *)arg;
	err = GLD_SUCCESS;
	
	NDIS_LOCK(sc);
	if (on) {
//		sc->ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		sc->ndis_filter |= NDIS_PACKET_TYPE_PROMISCUOUS;
	}else
		sc->ndis_filter &= ~NDIS_PACKET_TYPE_PROMISCUOUS;

	len = sizeof(sc->ndis_filter);
	error = ndis_set_info(sc, OID_GEN_CURRENT_PACKET_FILTER, &sc->ndis_filter, &len);
	if (error){
		NDISDBG((CE_CONT,"GLD_MAC_PROMISC_MULTI setting failed\n"));
		err = GLD_FAILURE;
	}
	NDIS_UNLOCK(sc);
	return (err);
}


static int
ndis_m_saddr(void *arg, const uint8_t *macaddr)
{
	NDISDBG((CE_CONT,"SADDR Not Support yet\n"));
	return (GLD_SUCCESS);
}

int skip_gstat=0;

static int
ndis_m_gstat(void *arg, uint_t statitem, uint64_t *val)
{
	int error, len=0;
	uint32_t speed=0;
	int rval = 0;
	struct ndis_80211_stats s;
	struct ndis_softc *sc = (struct ndis_softc *)arg;

	if(statitem == MAC_STAT_IFSPEED){
		ieee80211com_t *isc = (ieee80211com_t *)&(sc->ic);
		NDIS_LOCK(sc);
		if (NDIS_INITIALIZED(sc) && (isc->isc_state == IEEE80211_S_RUN)) {
		    len = sizeof(speed);
		    rval = ndis_get_info(sc, OID_GEN_LINK_SPEED, &speed, &len);
		    if (rval){ 
			NDISDBG((CE_CONT,"get link speed failed: %d\n", rval));
			speed = 0;
		    }
		    *val = speed*100;
		    NDISDBG((CE_CONT,"current link speed: %d\n", (int)*val));
		} else {
		    *val = 0;
		    NDISDBG((CE_CONT,"get link speed failed (not running)\n"));
		}
		NDIS_UNLOCK(sc);
		return 0;
	}else
		return(ENOTSUP);

	NDIS_LOCK(sc);
	if(skip_gstat){
		cmn_err(CE_CONT,"stop,waiting...\n");
		return(ENOTSUP);
	}else
		skip_gstat=0x1;

	if (!NDIS_INITIALIZED(sc)){
		NDISDBG((CE_WARN,"need to plumb\n"));
		skip_gstat = 0x0;
		NDIS_UNLOCK(sc);
		return(ENOTSUP);
	}

	len = sizeof(struct ndis_80211_stats);
	memset((uint8_t *)&s, 0, len);
	error = ndis_get_info(sc, OID_802_11_STATISTICS, (void *)&s, &len);
	if (error) {
		cmn_err(CE_CONT,"failed to get statistics %d\n", error);
		skip_gstat = 0x0;
		NDIS_UNLOCK(sc);
		return (ENOTSUP);
	}else
		NDISDBG((CE_CONT,"successfully gets statistics\n"));

	switch (statitem) {
	case MAC_STAT_OPACKETS:
		*val = s.TransmittedFragmentCount;
		break;
	case WIFI_STAT_MCAST_TX:
		*val = s.MulticastTransmittedFrameCount;
		break;
	case WIFI_STAT_TX_FAILED:
		*val = s.FailedCount;
		break;
	case WIFI_STAT_TX_RETRANS:
		*val = s.RetryCount;
		break;
	case WIFI_STAT_RTS_SUCCESS:
		*val = s.RTSSuccessCount;
		break;
	case WIFI_STAT_ACK_FAILURE:
		*val = s.ACKFailureCount;
		break;
	case MAC_STAT_IPACKETS:
		*val = s.ReceivedFragmentCount;
		break;
	case WIFI_STAT_MCAST_RX:
		*val = s.MulticastReceivedFrameCount;
		break;
	case WIFI_STAT_RTS_FAILURE:
		*val = s.RTSFailureCount;	
		break;
	case WIFI_STAT_RX_DUPS:
		*val = s.FrameDuplicateCount;
		break;
	case WIFI_STAT_FCS_ERRORS:
		*val = s.FCSErrorCount;
		break;
		
	/* Not support by me*/
	case WIFI_STAT_RX_FRAGS:
	case WIFI_STAT_TX_FRAGS:
	case WIFI_STAT_WEP_ERRORS:
	case MAC_STAT_OBYTES:
	case MAC_STAT_RBYTES:
	case MAC_STAT_NOXMTBUF:
	case MAC_STAT_IERRORS:
	case MAC_STAT_OERRORS:
		skip_gstat = 0x0;
		NDIS_UNLOCK(sc);
		return (ENOTSUP);
	default:
		skip_gstat = 0x0;
		NDIS_UNLOCK(sc);
		return (ENOTSUP);
	}

	skip_gstat = 0x0;
	NDIS_UNLOCK(sc);
	return (0);
}

static mblk_t *
ndis_m_tx(void *arg, mblk_t *mp)
{
	struct ndis_softc *sc = (struct ndis_softc *)arg;
	mblk_t *next;

	NDIS_LOCK(sc);
	if(sc->ndis_suspended) {
	    return (mp);
	}
	if(!NDIS_INITIALIZED(sc) || (sc->ndis_link == 0)){
		NDIS_UNLOCK(sc);
		NDISDBG((CE_CONT,"ndis_m_tx : media not ready\n"));
		return(mp);		
	}

	while (mp != NULL) {
		next = mp->b_next;
		mp->b_next = NULL;

		if (ndis_tx_begin(sc, mp)!= 0) {
			mp->b_next = next;
			break;
		}

		mp = next;
	}
	NDIS_UNLOCK(sc);
	
	return (mp);
}

int skip_start=0;

static int32_t
ndis_m_start(void *arg)
{
	struct ndis_softc	*sc = (struct ndis_softc *)arg;
	int			i, error;

	NDIS_LOCK(sc);
	if(skip_start){
		NDIS_UNLOCK(sc);
		return(GLD_SUCCESS);
	}else{
		skip_start = 0x1;
	}
		
	/*
	 * Avoid reintializing the link unnecessarily.
	 * This should be dealt with in a better way by
	 * fixing the upper layer modules so they don't
	 * call ifp->if_init() quite as often.
	 */
	if (NDIS_INITIALIZED(sc)){
		skip_start = 0;
		NDIS_UNLOCK(sc);
		return(GLD_SUCCESS);
	}
	/*
	 * Cancel pending I/O and free all RX/TX buffers.
	 */
	NDIS_UNLOCK(sc);
	ndis_m_stop(sc);
	if (ndis_init_nic(sc)){
		skip_start = 0;
		return(GLD_FAILURE);
	}
	/* Program the packet filter */

	sc->ndis_filter = NDIS_PACKET_TYPE_DIRECTED;

	sc->ndis_filter |= NDIS_PACKET_TYPE_BROADCAST;
	
	sc->ndis_filter &= ~NDIS_PACKET_TYPE_PROMISCUOUS;
		
	i = sizeof(sc->ndis_filter);

	NDIS_LOCK(sc);
	error = ndis_set_info(sc, OID_GEN_CURRENT_PACKET_FILTER,
	    (void *)&sc->ndis_filter, &i);

	if (error){
		cmn_err(CE_CONT, "set filter failed: %d\n", error);
		skip_start = 0;
		NDIS_UNLOCK(sc);
		return(GLD_FAILURE);
	}

	tickstop = 0;
	sc->ndis_txidx = 0;
	sc->ndis_txpending = sc->ndis_maxpkts;
	sc->ndis_link = 0;
        sc->ic.isc_state = IEEE80211_S_INIT;
        mac_link_update(sc->ndis_mh, LINK_STATE_DOWN);   

	//drv_usecwait(50);
	skip_start = 0;
	NDIS_UNLOCK(sc);

	/* Enable interrupts. */
	ndis_enable_intr(sc);
	/*
	 * Some drivers don't set this value. The NDIS spec says
	 * the default checkforhang timeout is "approximately 2
	 * seconds." We use 3 seconds, because it seems for some
	 * drivers, exactly 2 seconds is too fast.
	 */

	sc->ndis_block->nmb_checkforhangsecs = 3;
	tickstop = 0;
	if(sc->ndis_stat_ch == NULL){	
		ndis_tick(sc);
	}
	return(GLD_SUCCESS);
}


/*
 * Stop the adapter and free any mbufs allocated to the
 * RX and TX lists.
 */
//extern int ndis_exit;

static void
ndis_m_stop(void *arg)
{
	struct ndis_softc *sc =(struct ndis_softc *)arg ;
	int rval;
	uint32_t value=0;
	int len=4;
	ndis_80211_ssid ssid;
	
	NDIS_LOCK(sc);
	rval = ndis_set_info(sc, OID_802_11_DISASSOCIATE, (void *)&value, &len);
	if(rval)
		NDISDBG((CE_CONT,"in stop disassociate failed\n"));

	len = sizeof(ssid);
	memset(&ssid.ns_ssid[0], 0xff, 32);
	ssid.ns_ssidlen = 32;
	rval = ndis_set_info(sc, OID_802_11_SSID, (void *)&ssid, &len);
	if (rval)
		NDISDBG((CE_CONT,"set ssid failed: %d\n", rval));
	
	if(sc->ndis_stat_ch){
		untimeout(sc->ndis_stat_ch);
		tickstop = 0x1;
		sc->ndis_stat_ch = NULL;
	}

	while(identify)
		delay(20);
	NDIS_UNLOCK(sc);
	ndis_halt_nic(sc);
	NDIS_LOCK(sc);
	sc->ndis_block->nmb_pendingreq = NULL;
	sc->ndis_link = 0;
	sc->ic.isc_state = IEEE80211_S_INIT;
        mac_link_update(sc->ndis_mh, LINK_STATE_DOWN);   
	NDIS_UNLOCK(sc);
	return;
}

mac_callbacks_t ndis_m_callbacks = {
	MC_IOCTL,
	ndis_m_gstat,
	ndis_m_start,
	ndis_m_stop,
	ndis_m_prom,
	ndis_m_setmulti,
	ndis_m_saddr,
	ndis_m_tx,
	NULL,
	ndis_m_ioctl,
	NULL
};


/************************************************************
*
* All stuff for i80211. 
*
*************************************************************/

uint32_t
ieee80211_ieee2mhz(uint32_t chan, uint32_t  flags)
{
	if (flags & IEEE80211_CHAN_2GHZ) {	/* 2GHz band */
		if (chan == 14)
			return 2484;
		if (chan < 14)
			return 2407 + chan*5;
		else
			return 2512 + ((chan-15)*20);
	} else if (flags & IEEE80211_CHAN_5GHZ) {/* 5Ghz band */
		return 5000 + (chan*5);
	} else {				/* either, guess */
		if (chan == 14)
			return 2484;
		if (chan < 14)			/* 0-13 */
			return 2407 + chan*5;
		if (chan < 27)			/* 15-26 */
			return 2512 + ((chan-15)*20);
		return 5000 + (chan*5);
	}
}

/*
 * Convert channel to IEEE channel number.
 */
uint32_t
ieee80211_chan2ieee(ieee80211com_t *isc, struct ieee80211channel *ch)
{
	if (isc->isc_channels <= ch &&
	    ch <= &isc->isc_channels[IEEE80211_CHAN_MAX])
		return (ch - isc->isc_channels);
	else if (ch == IEEE80211_CHAN_ANYC)
		return (IEEE80211_CHAN_ANY);
	else {
		NDISDBG((CE_CONT,"wlan: invalid channel freq %u flags %x\n",
		    ch->ich_freq, ch->ich_flags));
		return (0);		/* XXX */
	}
}

/*
 * Convert MHz frequency to IEEE channel number.
 */
uint32_t
ieee80211_mhz2ieee(uint32_t freq, uint32_t flags)
{
	if (flags & IEEE80211_CHAN_2GHZ) {	/* 2GHz band */
		if (freq == 2484)
			return (14);
		if (freq < 2484)
			return ((freq - 2407) / 5);
		else
			return (15 + ((freq - 2512) / 20));
	} else if (flags & IEEE80211_CHAN_5GHZ) {	/* 5Ghz band */
		return ((freq - 5000) / 5);
	} else {				/* either, guess */
		if (freq == 2484)
			return (14);
		if (freq < 2484)
			return ((freq - 2407) / 5);
		if (freq < 5000)
			return (15 + ((freq - 2512) / 20));
		return ((freq - 5000) / 5);
	}
}

/* regs access attributes */
static ddi_device_acc_attr_t ndis_reg_accattr = {
	DDI_DEVICE_ATTR_V0,
	DDI_STRUCTURE_LE_ACC,
	DDI_STRICTORDER_ACC,
	DDI_DEFAULT_ACC
};
/*
 * Probe for an NDIS device. Check the PCI vendor and device
 * IDs against our list and return a device name if we find a match.
 */
static int
ndis_probe_pci(dev_info_t *dev, struct ndis_softc *sc)
{
	struct ndis_pci_type	*t;
	driver_object		*drv;
	vm_offset_t		img;
	uint32_t		tmp1,tmp2,tmp3;
	uint32_t		i=0;
	caddr_t			cfg_base;

	t = &ndis_devs[0];
	img = (vm_offset_t)drv_data_inuse;
	drv = windrv_lookup(img);

	if (drv == NULL)
		return (ENXIO);

	if(ddi_regs_map_setup(sc->devinfo_this, 0,&cfg_base,0,256,
		&ndis_reg_accattr,&sc->sc_cfg_handle) != DDI_SUCCESS){
		cmn_err(CE_CONT,"PCI Configure Head Setup failed\n");
		return (DDI_FAILURE);
	}

	while (t->ndis_name != NULL) {
		if (((tmp1 = ddi_get16(sc->sc_cfg_handle,(uint16_t *)(cfg_base + PCI_CONF_VENID))) 
			== t->ndis_vid) &&
		    ((tmp2 = ddi_get16(sc->sc_cfg_handle,(uint16_t *)(cfg_base + PCI_CONF_DEVID))) 
			== t->ndis_did) 
		     ){
			NDISINF((CE_CONT,"RealVendor = 0x%x,RealDevice = 0x%x,RealSubsys = 0x%x\n",
				tmp1,tmp2,tmp3));
			NDISINF((CE_CONT,"InfVendor = 0x%x,InfDevice= 0x%x, InfSubsys = 0x%x\n",
				t->ndis_vid,t->ndis_did,t->ndis_subsys));

			windrv_create_pdo(drv, dev);
			ddi_regs_map_free(&sc->sc_cfg_handle);	
			return (0);
		}
		i++;
		t = &ndis_devs[i];
	}
	ddi_regs_map_free(&sc->sc_cfg_handle);	
	return (ENXIO);
}

static int
ndis_attach_pci(struct ndis_softc *sc)
{
	struct ndis_pci_type	*t;
	int			devidx = 0, defidx = 0;
	dev_info_t		*dev;
	caddr_t		cfg_base;
	uint16_t	cmd;

	t = ndis_devs;
	
	if(ddi_regs_map_setup(sc->devinfo_this, 0,&cfg_base,0,256,
		&ndis_reg_accattr,&sc->sc_cfg_handle)
		!= DDI_SUCCESS)
		return (DDI_FAILURE);

	ddi_put16(sc->sc_cfg_handle, (uint16_t *)(cfg_base + 0x4),
		PCI_COMM_MAE|PCI_COMM_ME|PCI_COMM_IO);

	while(t->ndis_name != NULL) {
		if ((ddi_get16(sc->sc_cfg_handle,(uint16_t *)(cfg_base + PCI_CONF_VENID)) 
			== t->ndis_vid) &&
		    (ddi_get16(sc->sc_cfg_handle,(uint16_t *)(cfg_base + PCI_CONF_DEVID)) 
			== t->ndis_did)) {
			if (t->ndis_subsys == 0)
				defidx = devidx;
			else {
				if (t->ndis_subsys == 
					ddi_get32(sc->sc_cfg_handle,
					(uint32_t *)(cfg_base+PCI_CONF_SUBVENID)));
					break;
			}
		}
		t++;
		devidx++;
	}

	if (ndis_devs[devidx].ndis_name == NULL)
		sc->ndis_devidx = defidx;
	else
		sc->ndis_devidx = devidx;

	sc->ndis_iftype = PCIBus;
	ddi_regs_map_free(&sc->sc_cfg_handle);
	return(DDI_SUCCESS);
}

/*
 * Attach the interface. Allocate softc structures, do ifmedia
 * setup attach.
 */

int
ndis_attach(dev_info_t *devinfo, ddi_attach_cmd_t cmd)
{
	uint8_t			eaddr[ETHER_ADDR_LEN];
	void			*img;
	int			len;
	int			i,nb;
	char			strbuf[32];
	struct ndis_softc	*sc;
	int32_t 		instance;
	ddi_acc_handle_t 	handlex;
	driver_object		*drv;
	device_object		*pdo;
	uint32_t err;
	mac_register_t	*macp;
  	wifi_data_t		wd = { 0 };

	switch (cmd) {
	case DDI_RESUME:
	    if(ndisdrv_loaded > 0) {
		sc = ddi_get_soft_state(ndis_soft_state_p, 
		    ddi_get_instance(devinfo));
		if(sc->ndis_suspended) {
		    skip_start = 0;
		    ndis_reset_nic(sc);
		    ndis_m_start(sc);
		    sc->ndis_suspended = 0;
		}
	    }
	    return (DDI_SUCCESS);
	case DDI_ATTACH:
	    if (ndisdrv_loaded > 0)
		return 0;
	    ndisdrv_loaded++;
	    break;
	default:
	    return (DDI_FAILURE);
	}

	drv_data_inuse = kmem_zalloc(drv_data_len, KM_NOSLEEP);
	bcopy((char *)drv_data, (char *)drv_data_inuse,drv_data_len); 
	windrv_load(1, (vm_offset_t)drv_data_inuse, 0);
	windrv_wrap((funcptr)ndis_rxeof, &ndis_rxeof_wrap);
	windrv_wrap((funcptr)ndis_txeof, &ndis_txeof_wrap);
	windrv_wrap((funcptr)ndis_linksts, &ndis_linksts_wrap);
	windrv_wrap((funcptr)ndis_linksts_done,&ndis_linksts_done_wrap);

        ndis_debug = ddi_prop_get_int(DDI_DEV_T_ANY, devinfo, DDI_PROP_DONTPASS,"debug-level",0);

	instance = ddi_get_instance(devinfo);

	if (ddi_soft_state_zalloc(ndis_soft_state_p,
	    ddi_get_instance(devinfo)) != DDI_SUCCESS) {
		NDISINF((CE_WARN,("ndis_attach: Unable to alloc softstate\n")));
		goto failure_1;
	}
	sc = ddi_get_soft_state(ndis_soft_state_p, ddi_get_instance(devinfo));
	sc->devinfo_this= devinfo;
	ndis_attach_pci(sc);
	if(ndis_probe_pci(devinfo,sc) != 0){
		NDISINF((CE_CONT,"probe pci failed use 0\n"));
		goto failure_2;		
	}

	mutex_init(&sc->ndis_mtx, NULL, MUTEX_DRIVER, NULL);

	if (ddi_get_iblock_cookie(devinfo, 0, &sc->sc_iblock)
	    != DDI_SUCCESS) {
		NDISINF((CE_WARN,"ndis_attach: Can not get iblock cookie for INT\n"));
		goto failure_3;
	}

	mutex_init(&sc->ndis_intrmtx, NULL, MUTEX_DRIVER,(void *)sc->sc_iblock);

	if (ddi_add_intr(devinfo, 0, &sc->sc_iblock, 0, ndis_intr, (caddr_t)sc) != DDI_SUCCESS) {
		NDISINF((CE_WARN,"ndis_attach: Can not set intr for if_ndis driver\n"));
		goto failure_4;
	}

	sc->ndis_regvals = ndis_regvals;
	/* Create sysctl registry nodes */
	if(ndis_create_sysctls(sc) == EINVAL)
	{
		NDISINF((CE_CONT,"create sysctls failed\n"));
		goto failure_5;
	}
	img = drv_data_inuse;
	drv = windrv_lookup((vm_offset_t)img);
	pdo = windrv_find_pdo(drv, devinfo);

	/* Set up driver image in memory. */
	if (NdisAddDevice(drv, pdo, sc) != STATUS_SUCCESS)
	{
		NDISINF((CE_CONT,"load driver failed\n"));
		goto failure_6;
	}

	/* Tell the user what version of the API the driver is using. */
	cmn_err(CE_CONT, "NDIS API version: %d.%d\n",
	    sc->ndis_chars->nmc_version_major,
	    sc->ndis_chars->nmc_version_minor);

	/* Do resource conversion. */
	if(ndis_convert_res(sc) != DDI_SUCCESS)
	{
		NDISINF((CE_CONT,"convert resource failed\n"));
		goto failure_7;
	}

	/* Install our RX and TX interrupt handlers. */
	sc->ndis_block->nmb_senddone_func = ndis_txeof_wrap;
	sc->ndis_block->nmb_pktind_func = ndis_rxeof_wrap;
	
	/* Call driver's init routine. */
	if (ndis_init_nic(sc) != DDI_SUCCESS) {
		NDISINF((CE_CONT, "init handler failed\n"));
		goto failure_8;
	}

	cmn_err(CE_CONT,"%s: Congratulations, Attached Okay!\n",__func__);
	/*
	 * Create minor node of type DDI_NT_NET_WIFI
	 */
	(void) snprintf(strbuf, sizeof (strbuf), "%s%d",
	    NDIS_DRIVER_NAME, instance);
	err = ddi_create_minor_node(devinfo, strbuf, S_IFCHR,
	    instance + 1, DDI_NT_NET_WIFI, 0);
	if (err != DDI_SUCCESS)
		NDISDBG((CE_WARN,  "ndis_attach(): ddi_create_minor_node() failed\n"));

	/*
	 * Get station address from the driver.
	 */
	len = sizeof(eaddr);
	ndis_get_info(sc, OID_802_3_CURRENT_ADDRESS, &eaddr, &len);

	bcopy(eaddr, (char *)sc->ndis_mac_addr, ETHER_ADDR_LEN);
	cmn_err(CE_CONT,"Wlan MAC:%2x:%2x:%2x:%2x:%2x:%2x\n",
		eaddr[0],eaddr[1],eaddr[2],eaddr[3],eaddr[4],eaddr[5]);

	/*
	 * Figure out if we're allowed to use multipacket sends
	 * with this driver, and if so, how many.
	 */

	if (sc->ndis_chars->nmc_sendsingle_func &&
	    sc->ndis_chars->nmc_sendmulti_func == NULL) {
		sc->ndis_maxpkts = 1;
	} else {
		len = sizeof(sc->ndis_maxpkts);
		ndis_get_info(sc, OID_GEN_MAXIMUM_SEND_PACKETS,
		    &sc->ndis_maxpkts, &len);
	}
	NDISDBG((CE_CONT,"Maxpkts are %d\n",sc->ndis_maxpkts));
	sc->ndis_txpending =sc->ndis_maxpkts;

	sc->ndis_txarray = kmem_zalloc(sizeof(ndis_packet *) *
	    sc->ndis_maxpkts, KM_NOSLEEP);
	
	/* Allocate a pool of ndis_packets for TX encapsulation. */

	NdisAllocatePacketPool(&i, &sc->ndis_txpool,
	   sc->ndis_maxpkts+10, PROTOCOL_RESERVED_SIZE_IN_PACKET);

	if (i != NDIS_STATUS_SUCCESS) {
		sc->ndis_txpool = NULL;
		NDISINF((CE_CONT, "failed to allocate TX packet pool\n"));
		goto failure_9;
	}

	for(i = 0; i < sc->ndis_maxpkts;i++){
		int status;
		NdisAllocatePacket(&status, &sc->ndis_txarray[i], sc->ndis_txpool);
	}

	/*
	 * Provide initial settings for the WiFi plugin; whenever this
	 * information changes, we need to call mac_pdata_update()
	 */
  	wd.wd_secalloc = WIFI_SEC_NONE;
  	wd.wd_opmode = IEEE80211_M_STA;
	
	macp = mac_alloc(MAC_VERSION);
	if (macp == NULL) {
		NDISDBG((CE_NOTE, "%s: MAC version mismatch\n",__func__));
		goto failure_9;
	}

	macp->m_type_ident	= MAC_PLUGIN_IDENT_WIFI;
	macp->m_driver		= sc;
	macp->m_dip		= devinfo;
	macp->m_src_addr	= sc->ndis_mac_addr;
	macp->m_callbacks	= &ndis_m_callbacks;
	macp->m_min_sdu		= 0;
	macp->m_max_sdu		= IEEE80211_MTU;
  	macp->m_pdata		= &wd;
  	macp->m_pdata_size	= sizeof (wd);

	err = mac_register(macp, &sc->ndis_mh);
	mac_free(macp);
	if (err != 0) {
		NDISDBG((CE_NOTE, "%s: mac_register err\n",__func__));
		goto failure_10;
	}
	mac_link_update(sc->ndis_mh, LINK_STATE_DOWN);

	sc->ndis_oidcnt = 0;
	/* Get supported oid list. */
	ndis_get_supported_oids(sc, &sc->ndis_oids, &sc->ndis_oidcnt);

	/*
	 * See if the OID_802_11_CONFIGURATION OID is
	 * supported by this driver. If it is, then this an 802.11
	 * wireless driver, and we should set up media for wireless.
	 */
	for (i = 0; i < sc->ndis_oidcnt; i++) {
		if (sc->ndis_oids[i] == OID_802_11_CONFIGURATION) {
			sc->ndis_80211++;
			break;
		}
	}

	/* Do media setup */
	if (sc->ndis_80211) {
		struct ieee80211com	*ic = &sc->ic;
		ndis_80211_rates_ex	rates;
		struct ndis_80211_nettype_list *ntl;
		uint32_t		arg;
		int			r;

	        ic->isc_phytype = IEEE80211_T_DS;
		ic->isc_opmode = IEEE80211_M_STA;
		ic->isc_caps = IEEE80211_C_IBSS;
		ic->isc_state = IEEE80211_S_ASSOC;
		ic->isc_modecaps = (1<<IEEE80211_MODE_AUTO);
		len = 0;
		r = ndis_get_info(sc, OID_802_11_NETWORK_TYPES_SUPPORTED,
		    NULL, &len);
		if (r != ENOSPC)
			goto nonettypes;
		ntl = kmem_zalloc(len, KM_NOSLEEP);
		r = ndis_get_info(sc, OID_802_11_NETWORK_TYPES_SUPPORTED,
		    ntl, &len);
		if (r != 0) {
			kmem_free(ntl, len);
			goto nonettypes;
		}

		for (i = 0; i < ntl->ntl_items; i++) {
			switch (ntl->ntl_type[i]) {
			case NDIS_80211_NETTYPE_11FH:
			case NDIS_80211_NETTYPE_11DS:
				NDISDBG((CE_WARN,"Support 80211B \n"));
				ic->isc_modecaps |= (1<<IEEE80211_MODE_11B);
				break;
			case NDIS_80211_NETTYPE_11OFDM5:
				NDISDBG((CE_WARN,"Support 80211A \n"));
				ic->isc_modecaps |= (1<<IEEE80211_MODE_11A);
				break;
			case NDIS_80211_NETTYPE_11OFDM24:
				ic->isc_modecaps |= (1<<IEEE80211_MODE_11G);
				NDISDBG((CE_WARN,"Support 80211G\n"));
				break;
			default:
				NDISDBG((CE_WARN,"new nettype %d\n",ntl->ntl_type[i]));
				break;
			}
		}
		kmem_free(ntl, len);
nonettypes:
		len = sizeof(rates);
		bzero((char *)&rates, len);
		r = ndis_get_info(sc, OID_802_11_SUPPORTED_RATES,
		    (void *)rates, &len);
		if (r)
			NDISDBG((CE_WARN, "get rates failed: 0x%x\n", r));
		for(len = 0; len <16;  len++)
			NDISDBG((CE_WARN, "rate[len] = %d\n",rates[len]&0x7f));
		/*
		 * Since the supported rates only up to 8 can be supported,
		 * if this is not 802.11b we're just going to be faking it
		 * all up to heck.
		 */

#define TESTSETRATE(x, y)						\
	do {								\
		int			i;				\
		for (i = 0; i < ic->isc_sup_rates[x].rs_nrates; i++) {	\
			if (ic->isc_sup_rates[x].rs_rates[i] == (y))	\
				break;					\
		}							\
		if (i == ic->isc_sup_rates[x].rs_nrates) {		\
			ic->isc_sup_rates[x].rs_rates[i] = (y);		\
			ic->isc_sup_rates[x].rs_nrates++;		\
		}							\
	} while (0)

#define SETRATE(x, y)	\
	ic->isc_sup_rates[x].rs_rates[ic->isc_sup_rates[x].rs_nrates] = (y)
#define INCRATE(x)	\
	ic->isc_sup_rates[x].rs_nrates++

		ic->isc_curmode = IEEE80211_MODE_AUTO;
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11A))
			ic->isc_sup_rates[IEEE80211_MODE_11A].rs_nrates = 0;
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11B))
			ic->isc_sup_rates[IEEE80211_MODE_11B].rs_nrates = 0;
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11G))
			ic->isc_sup_rates[IEEE80211_MODE_11G].rs_nrates = 0;
		for (i = 0; i < len; i++) {
			switch (rates[i] & IEEE80211_RATE_VAL) {
			case 2:
			case 4:
			case 11:
			case 10:
			case 22:
				if (!(ic->isc_modecaps &
				    (1<<IEEE80211_MODE_11B))) {
					/* Lazy-init 802.11b. */
					ic->isc_modecaps |=
					    (1<<IEEE80211_MODE_11B);
					ic->isc_sup_rates[IEEE80211_MODE_11B].rs_nrates = 0;
				}
				SETRATE(IEEE80211_MODE_11B, rates[i]);
				INCRATE(IEEE80211_MODE_11B);
				break;
			default:
				if (ic->isc_modecaps & (1<<IEEE80211_MODE_11A)) {
					SETRATE(IEEE80211_MODE_11A, rates[i]);
					INCRATE(IEEE80211_MODE_11A);
				}
				if (ic->isc_modecaps & (1<<IEEE80211_MODE_11G)) {
					SETRATE(IEEE80211_MODE_11G, rates[i]);
					INCRATE(IEEE80211_MODE_11G);
				}
				break;
			}
		}

		/*
		 * If the hardware supports 802.11g, it most
		 * likely supports 802.11b and all of the
		 * 802.11b and 802.11g speeds, so maybe we can
		 * just cheat here.  Just how in the heck do
		 * we detect turbo modes, though?
		 */
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11B)) {
			TESTSETRATE(IEEE80211_MODE_11B,
			    IEEE80211_RATE_BASIC|2);
			TESTSETRATE(IEEE80211_MODE_11B,
			    IEEE80211_RATE_BASIC|4);
			TESTSETRATE(IEEE80211_MODE_11B,
			    IEEE80211_RATE_BASIC|11);
			TESTSETRATE(IEEE80211_MODE_11B,
			    IEEE80211_RATE_BASIC|22);
		}
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11G)) {
			TESTSETRATE(IEEE80211_MODE_11G, 47);
			TESTSETRATE(IEEE80211_MODE_11G, 72);
			TESTSETRATE(IEEE80211_MODE_11G, 96);
			TESTSETRATE(IEEE80211_MODE_11G, 108);
		}
		if (ic->isc_modecaps & (1<<IEEE80211_MODE_11A)) {
			TESTSETRATE(IEEE80211_MODE_11A, 47);
			TESTSETRATE(IEEE80211_MODE_11A, 72);
			TESTSETRATE(IEEE80211_MODE_11A, 96);
			TESTSETRATE(IEEE80211_MODE_11A, 108);
		}
#undef SETRATE
#undef INCRATE
		/*
		 * Taking yet more guesses here.
		 */
		for (i = 1; i < IEEE80211_CHAN_MAX; i++) {
			int chanflag = 0;

			if (ic->isc_sup_rates[IEEE80211_MODE_11G].rs_nrates)
				chanflag |= IEEE80211_CHAN_G;
			if (i <= 14)
				chanflag |= IEEE80211_CHAN_B;
			if (ic->isc_sup_rates[IEEE80211_MODE_11A].rs_nrates &&
			    i > 14)
				chanflag = IEEE80211_CHAN_A;
			if (chanflag == 0)
				break;
			ic->isc_channels[i].ich_freq =
			    ieee80211_ieee2mhz(i, chanflag);
			ic->isc_channels[i].ich_flags = chanflag;
		}

		i = sizeof(arg);
		r = ndis_get_info(sc, OID_802_11_WEP_STATUS, &arg, &i);
		if (arg != NDIS_80211_WEPSTAT_NOTSUPPORTED){
			ic->isc_caps |= IEEE80211_C_WEP;
			NDISDBG((CE_WARN,"wep support\n"));
		}
		i = sizeof(arg);
		r = ndis_get_info(sc, OID_802_11_POWER_MODE, &arg, &i);
		if (r == 0)
			ic->isc_caps |= IEEE80211_C_PMGT;
		bcopy(eaddr, (char *)sc->ndis_mac_addr, sizeof(eaddr));
		/* FIXME */
	//	ic->isc_ibss_chan = kmem_zalloc(sizeof(struct ieee80211channel),KM_NOSLEEP);
		ic->isc_ibss_chan = &ic->isc_channels[1];
	//	ic->isc_ibss_chan = IEEE80211_CHAN_ANYC;
		ic->isc_bss = kmem_zalloc(sizeof(struct ieee80211_node),KM_NOSLEEP);
		ic->isc_bss->ni_chan = ic->isc_ibss_chan;
	} 

	/* Override the status handler so we can detect link changes. */
	sc->ndis_block->nmb_status_func = ndis_linksts_wrap;
	sc->ndis_block->nmb_statusdone_func = ndis_linksts_done_wrap;
	
	/* We're done talking to the NIC for now; halt it. */
	ndis_halt_nic(sc);
	NDISDBG((CE_CONT,"Halt Wlan Adapter, Waiting for Plumb\n"));
	return (DDI_SUCCESS);

failure_10:
	(void) mac_unregister(sc->ndis_mh);
failure_9:
failure_8:
failure_7:
failure_6:
	ndis_flush_sysctls(sc);
failure_5:
	ddi_remove_intr(devinfo,0,sc->sc_iblock);
failure_4:
	mutex_destroy(&sc->ndis_intrmtx);
failure_3:	
	mutex_destroy(&sc->ndis_mtx);
failure_2:
	ddi_soft_state_free(ndis_soft_state_p, ddi_get_instance(devinfo));
failure_1:
	kmem_free(drv_data_inuse, drv_data_len);
	return(DDI_FAILURE);
}

/*
 * Shutdown hardware and free up resources. This can be called any
 * time after the mutex has been initialized. It is called in both
 * the error case in attach and the normal detach case so it needs
 * to be careful about only freeing resources that have actually been
 * allocated.
 */

static int
ndis_detach(dev_info_t *dip, ddi_detach_cmd_t cmd)
{
	struct ndis_softc	*sc;
	int ret;
	int i,test = 5;
	ieee80211com_t *isc;

	sc = ddi_get_soft_state(ndis_soft_state_p, ddi_get_instance(dip));
	switch (cmd) {
	case DDI_SUSPEND: 
		if ((ndisdrv_loaded > 0) && NDIS_INITIALIZED(sc)) {
		    isc = (ieee80211com_t *)&(sc->ic);
		    if (IEEE80211_S_RUN == isc->isc_state) {
			NDIS_LOCK(sc);
			sc->ndis_suspended = 1;
			(void) ndis_cfg_disconnect(sc, NULL, WLAN_COMMAND);
			NDIS_UNLOCK(sc);
			ndis_m_stop(sc);
		    }
		}
		return(DDI_SUCCESS);

	case DDI_DETACH:
		NDIS_LOCK(sc);
		ret = mac_unregister(sc->ndis_mh);
		if (ret) {
			NDIS_UNLOCK(sc);
			return (ret);
		}
		if(sc->sc_mcast)
			kmem_free(sc->sc_mcast,sc->sc_mcast_len);
		if (sc->ndis_txarray)
			kmem_free(sc->ndis_txarray, sizeof(ndis_packet *) *
	   			sc->ndis_maxpkts);
		ndis_unload_driver(sc);
		ExFreePool(sc->ndis_txpool);
		ddi_remove_intr(dip,0,sc->sc_iblock);
		sc->ndis_suspended = 1; /* just in case an ioctl is pending */
		NDIS_UNLOCK(sc);
		delay(50);
		mutex_destroy(&sc->ndis_mtx);
freelock:
		NDISDBG((CE_CONT,"Free mutex intrmtx\n"));
		if(mutex_tryenter(&sc->ndis_intrmtx) == 0){
			mutex_exit(&sc->ndis_intrmtx);
			delay(5);
			if(test--)
				goto freelock;
		}else{
			delay(10);
			goto freelock;
		}
		mutex_destroy(&sc->ndis_intrmtx);
		windrv_unload(1, (vm_offset_t)drv_data_inuse, 0);

		ndis_flush_sysctls(sc);		

		kmem_free((char *)drv_data_inuse, drv_data_len);
		windrv_unwrap(ndis_rxeof_wrap);
		windrv_unwrap(ndis_txeof_wrap);
		windrv_unwrap(ndis_linksts_wrap);
		windrv_unwrap(ndis_linksts_done_wrap);
		ddi_remove_minor_node(dip, NULL);
		ddi_soft_state_free(ndis_soft_state_p, ddi_get_instance(dip));
		ndisdrv_loaded--;
		sc->ndis_suspended = 0;
		return(DDI_SUCCESS);
	default:
		return (DDI_FAILURE);
	}
}

static int
ndis_quiesce(dev_info_t *dip)
{
	struct ndis_softc	*sc;
	ieee80211com_t *isc;

	sc = ddi_get_soft_state(ndis_soft_state_p, ddi_get_instance(dip));
	if ((ndisdrv_loaded > 0) && NDIS_INITIALIZED(sc)) {
	    isc = (ieee80211com_t *)&(sc->ic);
	    if (IEEE80211_S_RUN == isc->isc_state)
		ndis_m_stop(sc);
	}
	return(DDI_SUCCESS);
}

/*
 * A frame has been uploaded: pass the resulting mbuf chain up to
 * the higher level protocols.
 *
 * When handling received NDIS packets, the 'status' field in the
 * out-of-band portion of the ndis_packet has special meaning. In the
 * most common case, the underlying NDIS driver will set this field
 * to NDIS_STATUS_SUCCESS, which indicates that it's ok for us to
 * take posession of it. We then change the status field to
 * NDIS_STATUS_PENDING to tell the driver that we now own the packet,
 * and that we will return it at some point in the future via the
 * return packet handler.
 *
 * If the driver hands us a packet with a status of NDIS_STATUS_RESOURCES,
 * this means the driver is running out of packet/buffer resources and
 * wants to maintain ownership of the packet. In this case, we have to
 * copy the packet data into local storage and let the driver keep the
 * packet.
 */
__stdcall static void
ndis_rxeof(adapter, packets, pktcnt)
	ndis_handle		adapter;
	ndis_packet		**packets;
	uint32_t		pktcnt;
{
	struct ndis_softc	*sc;
	ndis_miniport_block	*block;
	ndis_packet		*p;
	uint32_t		flags;
	mblk_t		*m0, *m=NULL;
	int			i;
	wifi_data_t	wd={0};
	
	block = (ndis_miniport_block *)adapter;
	sc = (struct ndis_softc *)(block->nmb_ifp);

//	NDISDBG((CE_CONT, "ndis_rxeof,pcount %d\n",pktcnt));
	for (i = 0; i < pktcnt; i++) {
		p = packets[i];
		/* Stash the softc here so ptom can use it. */
		p->np_softc = sc;
		if (ndis_ptom(&m0, p)) {
			NDISINF((CE_CONT,"ptom failed\n"));
			if (p->np_oob.npo_status == NDIS_STATUS_SUCCESS)
				ndis_return_packet(sc, p);
		} else {
				rxcnt++;
				if (p->np_oob.npo_status == NDIS_STATUS_RESOURCES) 
					cmn_err(CE_CONT,"RXOF resource err\n");
				else
					p->np_oob.npo_status = NDIS_STATUS_PENDING;
				bcopy(&g_wd,&wd,sizeof(wifi_data_t));
				m = mac_wifi_header_fake(m0, (void *)&wd);
				if(m == NULL){
					cmn_err(CE_CONT,"Null package!!!\n");
				}else{	
					uploadcnt++;
					mac_rx(sc->ndis_mh, NULL, m);
					NDISDBG((CE_CONT, "mac_rx done! \n"));
				}
		}
	}
	return;
}

/*
 * A frame was downloaded to the chip. It's safe for us to clean up
 * the list buffers.
 */
__stdcall static void
ndis_txeof(adapter, packet, status)
	ndis_handle		adapter;
	ndis_packet		*packet;
	ndis_status		status;

{
	struct ndis_softc	*sc;
	ndis_miniport_block	*block;
	int			idx;
	mblk_t		*m;

	//NDISDBG((CE_CONT, "ndis_txeof\n"));
	block = (ndis_miniport_block *)adapter;
	sc = (struct ndis_softc *)block->nmb_ifp;

	NDIS_LOCK(sc);
	m = packet->np_m0;
	idx = packet->np_txidx;
	//ndis_free_packet(packet);
	ndis_free_bufs(packet->np_private.npp_head);
	packet->np_private.npp_head = NULL;
	//NdisFreePacket(packet);
	freemsg(m);
	packet->np_m0 = NULL;

	//sc->ndis_txarray[idx] = NULL;
	sc->ndis_txpending++;
	NDIS_UNLOCK(sc);
	if(sc->ndis_txpending == 1)
		mac_tx_update(sc->ndis_mh);
	return;
}

__stdcall static void
ndis_linksts(adapter, status, sbuf, slen)
	ndis_handle		adapter;
	ndis_status		status;
	void			*sbuf;
	uint32_t		slen;
{
	ndis_miniport_block	*block;
	struct ndis_softc	*sc;

	block = adapter;
	sc = (struct ndis_softc *)block->nmb_ifp;

	block = adapter;
	block->nmb_getstat = status;

	return;
}

__stdcall static void
ndis_linksts_done(adapter)
	ndis_handle		adapter;
{
	ndis_miniport_block	*block;
	struct ndis_softc	*sc;

	block = adapter;
	sc = (struct ndis_softc *)block->nmb_ifp;

	NDISDBG((CE_CONT, "ndis_linksts_done\n"));

	//NDIS_LOCK(sc);
	if (!NDIS_INITIALIZED(sc)) {
	//	NDIS_UNLOCK(sc);
		return;
	}

	switch (block->nmb_getstat) {
	case NDIS_STATUS_MEDIA_CONNECT:
		ndis_sched(ndis_ticktask, sc, NDIS_TASKQUEUE);
		break;
	case NDIS_STATUS_MEDIA_DISCONNECT:
		if (sc->ndis_link)
			ndis_sched(ndis_ticktask, sc, NDIS_TASKQUEUE);
		break;
	default:
		break;
	}

	//NDIS_UNLOCK(sc);
	return;
}

static uint32_t
ndis_intr(caddr_t arg)
{
	struct ndis_softc	*sc;
	int			is_our_intr = 0;
	int			call_isr = 0;
	uint8_t		irql;
	ndis_miniport_interrupt	*intr;
	
	sc = ( struct ndis_softc *)arg;
	intr = sc->ndis_block->nmb_interrupt;
	
	if (sc->ndis_block->nmb_miniportadapterctx == NULL
		|| intr == NULL || sc->ndis_suspended){
		return(DDI_INTR_UNCLAIMED);
	}
	
	if (sc->ndis_block->nmb_interrupt->ni_isrreq == TRUE){
		ndis_isr(sc, &is_our_intr, &call_isr);
	}else {
		NDISINF((CE_CONT,"SO BAD\n"));
		ndis_disable_intr(sc);
		call_isr = 1;
	}
	if ((is_our_intr || call_isr)){
	//	ndis_intrhandler(sc);
		mutex_enter(&sc->ndis_intrmtx);
		IoRequestDpc(sc->ndis_block->nmb_deviceobj, NULL, sc);
		mutex_exit(&sc->ndis_intrmtx);
		return(DDI_INTR_CLAIMED);
	}
	return(DDI_INTR_UNCLAIMED);
}

static void
ndis_tick(xsc)
	void			*xsc;
{
	struct ndis_softc	*sc;
	sc = (struct ndis_softc	*)xsc;
	if (!NDIS_INITIALIZED(sc) || tickstop) {
		return;
	}
	ndis_sched(ndis_ticktask, sc, NDIS_TASKQUEUE);
	sc->ndis_stat_ch = timeout(ndis_tick, sc,
	    drv_usectohz(1000000) * sc->ndis_block->nmb_checkforhangsecs);
	return;
}


static void
ndis_ticktask(xsc)
	void			*xsc;
{
	struct ndis_softc	*sc;
	__stdcall ndis_checkforhang_handler hangfunc;
	uint8_t			rval;
	ndis_media_state	linkstate;
	int			error, len;
	uint8_t			irql;
	wifi_data_t 	wd;
	
	sc = xsc;

	if (!NDIS_INITIALIZED(sc)) {
		return;
	}

	hangfunc = sc->ndis_chars->nmc_checkhang_func;

	if(identify)
		return;

	NDIS_LOCK(sc);
	identify = 0x1;
	NDISDBG((CE_CONT,"TxFiFo slots [%d], Iqlen[%d]\n",
		sc->ndis_txpending, *ndis_i_pcnt));
	//cmn_err(CE_CONT,"rxeof %d, macrx %d\n",rxcnt, uploadcnt);
	rxcnt = 0; uploadcnt = 0;
	if (hangfunc != NULL) {
		cmn_err(CE_CONT,"hangcheck\n");
		rval = MSCALL1(hangfunc,
		    sc->ndis_block->nmb_miniportadapterctx);
		if (rval == TRUE) {
			cmn_err(CE_CONT,"reset it\n");
			ndis_reset_nic(sc);
			NDIS_UNLOCK(sc);
			return;
		}
	}
	
	len = sizeof(linkstate);
	error = ndis_get_info(sc, OID_GEN_MEDIA_CONNECT_STATUS,
	    (void *)&linkstate, &len);

	if (sc->ndis_link == 0 && linkstate == nmc_connected) {
		cmn_err(CE_CONT, "link up\n");
		sc->ndis_link = 1;
		sc->ic.isc_state = IEEE80211_S_RUN;
		if (sc->ndis_80211){
			ndis_getstate_80211(sc);		
			mac_link_update(sc->ndis_mh, LINK_STATE_UP);
			bcopy(&g_wd, &wd, sizeof(wd));
			(void) mac_pdata_update(sc->ndis_mh, &wd, sizeof (wd));
		}
	}else if (sc->ndis_link == 1 && linkstate == nmc_disconnected) {
		cmn_err(CE_CONT, "link down\n");
		sc->ndis_link = 0;
		sc->ic.isc_state = IEEE80211_S_ASSOC;
		mac_link_update(sc->ndis_mh, LINK_STATE_DOWN);
	}
	
	identify = 0;
	NDIS_UNLOCK(sc);
	return;
}

static void
ndis_setstate_80211(sc)
	struct ndis_softc	*sc;
{
	struct ieee80211com	*ic;
	ndis_80211_ssid		ssid;
	ndis_80211_config	config;
	ndis_80211_wep		wep;
	int			i, rval = 0, len;
	uint32_t		arg;

	ic = &sc->ic;
	NDISDBG((CE_CONT,"%s\n",__func__));

	if (!NDIS_INITIALIZED(sc))
		return;

	/* Set network infrastructure mode. */

	len = sizeof(arg);
	if (ic->isc_opmode == IEEE80211_M_IBSS){
		arg = NDIS_80211_NET_INFRA_IBSS;
		NDISDBG((CE_CONT,"IBSS Mode\n"));
	}else{
		NDISDBG((CE_CONT,"STA Mode\n"));
		arg = NDIS_80211_NET_INFRA_BSS;
	}

	rval = ndis_set_info(sc, OID_802_11_INFRASTRUCTURE_MODE, (void *)&arg, &len);

	if (rval)
		NDISDBG((CE_CONT," %s: set infra failed: %d\n", __func__, rval));

	/* Set WEP */

	if (ic->isc_flags & IEEE80211_F_WEPON) {
		if(ic->isc_wep_txkey == 0 || ic->isc_wep_txkey > 4){
			NDISDBG((CE_CONT,"%s: anyway, set wepkeyindex to 0\n",__func__));
			ic->isc_wep_txkey = 0;
		}
		for (i = 0; i < IEEE80211_WEP_NKID; i++) {
			if (ic->isc_nw_keys[i].wk_len) {
				bzero((char *)&wep, sizeof(wep));
				wep.nw_keylen = ic->isc_nw_keys[i].wk_len;
				wep.nw_keyidx = i;
				wep.nw_length = (sizeof(uint32_t) * 3)
				    + wep.nw_keylen;
				if (i == ic->isc_wep_txkey){
					wep.nw_keyidx |= NDIS_80211_WEPKEY_TX;
					NDISDBG((CE_CONT,"%s: Active keyindex %d\n",
							__func__,i ));
				}
				bcopy(ic->isc_nw_keys[i].wk_key,
				    wep.nw_keydata, wep.nw_length);
				len = sizeof(wep);
				rval = ndis_set_info(sc,
				    OID_802_11_ADD_WEP, &wep, &len);
				if (rval)
					NDISDBG((CE_CONT,"set wepkey failed: %d\n", rval));
			}
		}
		arg = NDIS_80211_WEPSTAT_ENABLED;
		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_WEP_STATUS, (void *)&arg, &len);
		if (rval)
			NDISDBG((CE_CONT, "enable WEP failed: %d\n", rval));
		arg = NDIS_80211_PRIVFILT_8021XWEP;
		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_PRIVACY_FILTER, (void *)&arg, &len);
	} else {
		arg = NDIS_80211_WEPSTAT_DISABLED;
		len = sizeof(arg);
		rval = ndis_set_info(sc, OID_802_11_WEP_STATUS, (void *)&arg, &len);
		if (rval)
			NDISDBG((CE_CONT, "disable WEP failed: %d\n", rval));
	}

	arg = ic->isc_authmode;
	len = sizeof(arg);
	rval = ndis_set_info(sc, OID_802_11_AUTHENTICATION_MODE, 
						(void *)&arg, &len);
	if (rval)
		NDISINF((CE_CONT, 
			"set auth mode failed: %d\n", rval));
#if 0					
	len = sizeof(config);
	bzero((char *)&config, len);
	config.nc_length = len;
	config.nc_fhconfig.ncf_length = sizeof(ndis_80211_config_fh);
	rval = ndis_get_info(sc, OID_802_11_CONFIGURATION, &config, &len); 
	if (rval)
		NDISDBG((CE_CONT, "get configuration failed: %d\n", rval));

	/*
	 * Some drivers expect us to initialize these values, so
	 * provide some defaults.
	 */
	 
	if (config.nc_beaconperiod == 0)
		config.nc_beaconperiod = 100;
	if (config.nc_atimwin == 0)
		config.nc_atimwin = 100;
	if (config.nc_fhconfig.ncf_dwelltime == 0)
		config.nc_fhconfig.ncf_dwelltime = 200;

	if (rval == 0 ) { 
		int chan, chanflag;
		NDISDBG((CE_CONT,"isc_ibss_chan not any\n"));
		chan = ieee80211_chan2ieee(ic, ic->isc_ibss_chan);
		chanflag = config.nc_dsconfig > 2500000 ? IEEE80211_CHAN_2GHZ :
		    IEEE80211_CHAN_5GHZ;
		if (chan != ieee80211_mhz2ieee(config.nc_dsconfig / 1000, 0)) {
			config.nc_dsconfig =
			    ic->isc_ibss_chan->ich_freq * 1000;
			ic->isc_bss->ni_chan = ic->isc_ibss_chan;
			len = sizeof(config);
			config.nc_length = len;
			config.nc_fhconfig.ncf_length =
			    sizeof(ndis_80211_config_fh);
			rval = ndis_set_info(sc, OID_802_11_CONFIGURATION,
			    (void *)&config, &len);
			if (rval)
				NDISDBG((CE_CONT,"couldn't change "
				    "DS config to %ukHz: %d\n",
				    config.nc_dsconfig, rval));
		}
	}else if (rval)
		NDISDBG((CE_CONT,"couldn't retrieve channel info: %d\n", rval));
#endif //No need for bcm43xx

	/* Set SSID -- always do this last. */

	len = sizeof(ssid);
	bzero((char *)&ssid, len);
	ssid.ns_ssidlen = ic->isc_des_esslen;
	if (ssid.ns_ssidlen == 0) {
		ssid.ns_ssidlen = 1;
	} else
		bcopy(ic->isc_des_essid, ssid.ns_ssid, ssid.ns_ssidlen);
	
	NDISDBG((CE_CONT,"copy bssid=%s\n",ic->isc_des_essid));
	rval = ndis_set_info(sc, OID_802_11_SSID, (void *)&ssid, &len);
	if (rval)
		NDISDBG((CE_CONT,"set ssid failed setstate80211: %d\n", rval));

	return;
}

static int
ndis_get_assoc(	struct ndis_softc *sc, ndis_wlan_bssid_ex **assoc)
{
	ndis_80211_bssid_list_ex	*bl;
	ndis_wlan_bssid_ex	*bs;
	ndis_80211_macaddr	bssid;
	int i, len, error;

	NDISDBG((CE_CONT,"%s\n",__func__));
	if (!sc->ndis_link)
		return(ENOENT);

	len = sizeof(bssid);
	error = ndis_get_info(sc, OID_802_11_BSSID, (void *)&bssid, &len);
	if (error) {
		NDISDBG((CE_CONT,"assoc failed to get bssid\n"));
		return(ENOENT);
	}
/*	
	len = 0;
	error = ndis_get_info(sc, OID_802_11_BSSID_LIST, NULL, &len);
	if (error != ENOSPC) {
		NDISDBG((CE_CONT,"bssid_list failed\n"));
		return (error);
	}
*/
	len = 8192;
	bl = kmem_zalloc(len, KM_NOSLEEP);
	error = ndis_get_info(sc, OID_802_11_BSSID_LIST, (void *)bl, &len);
	if (error) {
		kmem_free(bl, 8192);
		NDISDBG((CE_CONT,"bssid_list failed\n"));
		return (error);
	}

	bs = (ndis_wlan_bssid_ex *)&bl->nblx_bssid[0];
	for (i = 0; i < bl->nblx_items; i++) {
		if (bcmp(bs->nwbx_macaddr, bssid, sizeof(bssid)) == 0) {
			*assoc = kmem_zalloc(bs->nwbx_len, KM_NOSLEEP);
			if (*assoc == NULL) {
				kmem_free(bl, 8192);
				return(ENOMEM);
			}
			bcopy((char *)bs, (char *)*assoc, bs->nwbx_len);
			bcopy((char *)bs->nwbx_macaddr, (char *)g_wd.wd_bssid,
				IEEE80211_ADDR_LEN);
			g_wd.wd_opmode = sc->ic.isc_opmode;
			if(sc->ic.isc_flags & IEEE80211_F_WEPON)
				g_wd.wd_secalloc = WIFI_SEC_WEP;
			else
				g_wd.wd_secalloc = WIFI_SEC_NONE;

			kmem_free(bl, 8192);
			return(0);
		}	
		bs = (ndis_wlan_bssid_ex *)((char *)bs + bs->nwbx_len);
	}

	kmem_free(bl, 8192);
	NDISINF((CE_CONT,"Associate Failed\n"));	
	return(ENOENT);
}

static void
ndis_getstate_80211(struct ndis_softc	*sc)
{
	struct ieee80211com	*ic;
	ndis_80211_ssid		ssid;
	ndis_80211_config	config;
	ndis_wlan_bssid_ex	*bs;
	int32_t			rval, len, i = 0;
	uint32_t		arg;
	wifi_data_t		wd = {0};

	ic = &sc->ic;

	NDISDBG((CE_CONT,"ndis_getstate_80211\n"));
	if (!NDIS_INITIALIZED(sc))
		return;

	/*
	 * If we're associated, retrieve info on the current bssid.
	 */
	if ((rval = ndis_get_assoc(sc, &bs)) == 0) {
		switch(bs->nwbx_nettype) {
		case NDIS_80211_NETTYPE_11FH:
		case NDIS_80211_NETTYPE_11DS:
			ic->isc_curmode = IEEE80211_MODE_11B;
			break;
		case NDIS_80211_NETTYPE_11OFDM5:
			ic->isc_curmode = IEEE80211_MODE_11A;
			break;
		case NDIS_80211_NETTYPE_11OFDM24:
			ic->isc_curmode = IEEE80211_MODE_11G;
			break;
		default:
			NDISDBG((CE_CONT, "unknown nettype %d\n", arg));
			break;
		}
		kmem_free(bs,bs->nwbx_len);
	} else
		return;

	len = sizeof(ssid);
	bzero((char *)&ssid, len);
	rval = ndis_get_info(sc, OID_802_11_SSID, &ssid, &len);

	if (rval)
		NDISDBG((CE_CONT,"get ssid failed: %d\n", rval));
	else{
		bcopy(ssid.ns_ssid, ic->isc_bss->ni_essid, ssid.ns_ssidlen);
		ic->isc_bss->ni_esslen = ssid.ns_ssidlen;
	}

	len = sizeof(arg);
	rval = ndis_get_info(sc, OID_GEN_LINK_SPEED, &arg, &len);
	if (rval)
		NDISDBG((CE_CONT,"get link speed failed: %d\n", rval));
	NDISDBG((CE_CONT, "LinkSpeed = %dMbps\n", arg/10000));	

	if (ic->isc_modecaps & (1<<IEEE80211_MODE_11B)) {
		ic->isc_bss->ni_rates = ic->isc_sup_rates[IEEE80211_MODE_11B];
		for (i = 0; i < ic->isc_bss->ni_rates.rs_nrates; i++) {
			if ((ic->isc_bss->ni_rates.rs_rates[i] &
			    IEEE80211_RATE_VAL) == arg / 5000)
				break;
		}
	}

	if (i == ic->isc_bss->ni_rates.rs_nrates &&
	    ic->isc_modecaps & (1<<IEEE80211_MODE_11G)) {
		ic->isc_bss->ni_rates = ic->isc_sup_rates[IEEE80211_MODE_11G];
		for (i = 0; i < ic->isc_bss->ni_rates.rs_nrates; i++) {
			if ((ic->isc_bss->ni_rates.rs_rates[i] &
			    IEEE80211_RATE_VAL) == arg / 5000)
				break;
		}
	}

	if (i == ic->isc_bss->ni_rates.rs_nrates)
		NDISDBG((CE_CONT,"no matching rate for: %d\n", arg / 5000));
	else
		ic->isc_bss->ni_txrate = i;

	if (ic->isc_caps & IEEE80211_C_PMGT) {
		len = sizeof(arg);
		rval = ndis_get_info(sc, OID_802_11_POWER_MODE, &arg, &len);

		if (rval)
			NDISDBG((CE_CONT, "get power mode failed: %d\n", rval));
		if (arg == NDIS_80211_POWERMODE_CAM)
			ic->isc_flags &= ~IEEE80211_F_PMGTON;
		else
			ic->isc_flags |= IEEE80211_F_PMGTON;
	}

	len = sizeof(config);
	bzero((char *)&config, len);
	config.nc_length = len;
	config.nc_fhconfig.ncf_length = sizeof(ndis_80211_config_fh);
	rval = ndis_get_info(sc, OID_802_11_CONFIGURATION, &config, &len);   
	if (rval == 0) { 
		int chan;

		chan = ieee80211_mhz2ieee(config.nc_dsconfig / 1000, 0);
		if (chan < 0 || chan >= IEEE80211_CHAN_MAX) {
				NDISDBG((CE_CONT,"current channel "
				    "(%uMHz) out of bounds\n", 
				    config.nc_dsconfig / 1000));
			ic->isc_bss->ni_chan = &ic->isc_channels[1];
		} else{
			ic->isc_bss->ni_chan = &ic->isc_channels[chan];
			NDISDBG((CE_CONT,"%s: connected channel is [%d]\n",__func__, chan));	
		}
	} else
		NDISDBG((CE_CONT,"couldn't retrieve "
		    "channel info: %d\n", rval));

	return;
}

